UNIT IK410_0;

{-------------------------------------------------------
  DR. JOHANNES HEIDENHAIN GmbH, Traunreut, Germany

  Driver Unit for IK410 (Basic Functions)

  V 1.00
  Okt 1997
 -----------------------------------------------------------}

{$N+,E+}
{$V+}
{$R+}

interface

PROCEDURE switch_z1(baseadr:word;z1:boolean);

PROCEDURE write_counter(baseadr:word;address:byte;datum:word);
FUNCTION  read_counter(baseadr:word;address:byte):word;
PROCEDURE wr_iic(baseadr:word;scl,sda:boolean);
FUNCTION  read_iic(baseadr:word):boolean;

PROCEDURE soft_l0(baseadr:word);
PROCEDURE soft_l1(baseadr:word);
FUNCTION  read_count_value48(baseadr:word;reg:byte):comp;
FUNCTION  read_count_value32(baseadr:word;reg:byte):comp;
FUNCTION  latched(baseadr:word;reg:byte):boolean;
PROCEDURE poll_latch(baseadr:word;reg:byte);

Implementation
{-------------------------------------------------------
	    switch_z1
 This procedure switches the analog multiplexer for the
 encoder signals.
 -------------------------------------------------------}
PROCEDURE switch_z1(baseadr:word;z1:boolean);
  BEGIN

    if z1 then
      begin
        portw[baseadr+$08]:=1;
        port[baseadr+$20]:=1;
      end
    else
      begin
        portw[baseadr+$08]:=1;
        port[baseadr+$20]:=0;
      end;
  END;
{-------------------------------------------------------
                      write_counter
 -------------------------------------------------------
 This procedure writes a value to the IK 410.

 This is a hardware-specific code. Write your own code
 for this function for access to the addresses of
 the hardware in which you have implemented the IK 410.
 -------------------------------------------------------}
PROCEDURE
 write_counter(baseadr:word;address:byte;datum:word);
  VAR adr_reg,adr_point,adr_gate : word;

  BEGIN
    baseadr:=baseadr and $0FE0;
    address:=address and $001F;

    (* Load address pointer into address register *)
    adr_reg:=baseadr or $0008;
    adr_point:=address shr 2;
    portw[adr_reg]:=adr_point;

    (* Calculate port address *)
    adr_gate:=baseadr or (address and $03);

    (* Write <datum> to the counter *)
    portw[adr_gate]:=datum;
  END;
{-------------------------------------------------------
                        read_counter
 -------------------------------------------------------
 This procedure reads a value from the IK 410.

 This is a hardware-specific code. Write your own code
 for this function for access to the addresses of
 the hardware in which you have implemented the IK 410.
 -------------------------------------------------------}
FUNCTION read_counter(baseadr:word;address:byte):word;
  VAR adr_reg,adr_point,adr_gate : word;

  BEGIN
    baseadr:=baseadr and $0FE0;
    address:=address and $001F;

    (* Load address pointer into address register *)
    adr_reg:=baseadr or $0008;
    adr_point:=address shr 2;
    portw[adr_reg]:=adr_point;

    (* Calculate port address *)
    adr_gate:=baseadr or (address and $03);

    (* Read datum from the counter *)
    read_counter:=portw[adr_gate];
  END;
{-------------------------------------------------------
		   wr_iic
 -------------------------------------------------------
 This procedure writes a bit to the iic bus
 -------------------------------------------------------}
PROCEDURE wr_iic(baseadr:word;scl,sda:boolean);
VAR
  adresse : word;

  BEGIN

    if scl then
      begin
        portw[baseadr+$08]:=0;
        portw[baseadr+$20]:=0;
      end
    else
      begin
        portw[baseadr+$08]:=0;
        portw[baseadr+$20]:=1;
      end;

    if sda then
      begin
        portw[baseadr+$08]:=0;
        portw[baseadr+$22]:=0;
      end
    else
      begin
        portw[baseadr+$08]:=0;
        portw[baseadr+$22]:=1;
      end;
  END;

{-------------------------------------------------------
		   read_iic
 -------------------------------------------------------
 This procedure reads a bit via the iic bus
 -------------------------------------------------------}
FUNCTION read_iic(baseadr:word):boolean;
VAR
  buffer : word;
  BEGIN
    buffer:=read_counter(baseadr,14);      (* Read SO *)
    buffer:=buffer and $0020;
    if buffer=$0020 then read_iic:=true else read_iic:=false;
  END;
{-------------------------------------------------------
                         soft_l0
 -------------------------------------------------------
 This procedure reads the measured value and stores
 it in data register 0.
 -------------------------------------------------------}
PROCEDURE soft_l0(baseadr:word);
  BEGIN
    write_counter(baseadr,14,$0001);
  END;
{-------------------------------------------------------
                         soft_l1
 -------------------------------------------------------
 This procedure reads the measured value and stores
 it in data register 1.
 -------------------------------------------------------}
PROCEDURE soft_l1(baseadr:word);
  BEGIN
    write_counter(baseadr,14,$0002);
  END;
{-------------------------------------------------------
                    read_count_value48
 -------------------------------------------------------
 This function reads 48 bits of a measured value.
 -------------------------------------------------------}
FUNCTION
 read_count_value48(baseadr:word;reg:byte):comp;
  TYPE
    vartype = (li,by);
    mapper = record
               case wert:vartype of
                 li : (field0:comp);
                 by : (field1:array[0..3] of word);
               end;
  VAR
    buffer : mapper;
  BEGIN
    case reg of
     0 : begin
           buffer.field1[0]:=read_counter(baseadr,0);
           buffer.field1[1]:=read_counter(baseadr,2);
           buffer.field1[2]:=read_counter(baseadr,4);
           if (buffer.field1[2] and $8000)=$8000 then
           buffer.field1[3]:=
                          $FFFF else buffer.field1[3]:=0;
         end;
     1 : begin
           buffer.field1[0]:=read_counter(baseadr,6);
           buffer.field1[1]:=read_counter(baseadr,8);
           buffer.field1[2]:=read_counter(baseadr,10);
           if (buffer.field1[2] and $8000)=$8000 then
           buffer.field1[3]:=
                          $FFFF else buffer.field1[3]:=0;
         end;
     end;
     read_count_value48:=buffer.field0;
  END;
{-------------------------------------------------------
  		    read_count_value32
 -------------------------------------------------------
 This function reads 32 bits of a measured value.
 -------------------------------------------------------}
FUNCTION
 read_count_value32(baseadr:word;reg:byte):comp;
  TYPE
    vartype = (li,by);
    mapper = record
               case wert:vartype of
                 li : (field0:longint);
                 by : (field1:array[0..1] of word);
               end;
  VAR
    buffer : mapper;
  BEGIN
    case reg of
     0 : begin
           buffer.field1[0]:=read_counter(baseadr,0);
           buffer.field1[1]:=read_counter(baseadr,2);
         end;
     1 : begin
           buffer.field1[0]:=read_counter(baseadr,6);
           buffer.field1[1]:=read_counter(baseadr,8);
         end;
     end;
     read_count_value32:=buffer.field0;
  END;
{-------------------------------------------------------
                         latched
 -------------------------------------------------------
 This function checks whether a measured value is
 latched in data register 0 or 1.
 -------------------------------------------------------}
FUNCTION latched(baseadr:word;reg:byte):boolean;
  BEGIN
      case reg of
        0: latched:=
        (read_counter(baseadr,14) and $0001 ) = $0001;
        1: latched:=
        (read_counter(baseadr,14) and $0002 ) = $0002;
      end;
  END;
{-------------------------------------------------------
                       poll_latch
 -------------------------------------------------------
 This function polls until a measured value is
 latched in data register 0 or 1.
 -------------------------------------------------------}
PROCEDURE poll_latch(baseadr:word;reg:byte);
  BEGIN
      case reg of
        0: begin
             repeat
             until latched(baseadr,0);
           end;
        1: begin
             repeat
             until latched(baseadr,1);
           end;
      end;
  END;
{-------------------------------------------------------}
END.