{-------------------------------------------------------}
{ DR. JOHANNES HEIDENHAIN GmbH, Traunreut, Germany      }
{                                                       }
{ Functions and Procedures to Adjust Pots of IK410      }
{                                                       }
{ V 1.00                                                }
{ Okt 1997                                              }
{-------------------------------------------------------}
{ Reg. 0 stores poti value                              }
{ Reg. 2, 3 stores offset value of counter chip         }
{-------------------------------------------------------}
(***************************************)
PROCEDURE scl0_sda0(board:IK410_pointr);
  BEGIN
    wr_iic(board^.baseadr,false,false);
  END;
(***************************************)
PROCEDURE scl1_sda0(board:IK410_pointr);
  BEGIN
    wr_iic(board^.baseadr,true,false);
  END;
(***************************************)
PROCEDURE scl0_sda1(board:IK410_pointr);
  BEGIN
    wr_iic(board^.baseadr,false,true);
  END;
(***************************************)
PROCEDURE scl1_sda1(board:IK410_pointr);
  BEGIN
    wr_iic(board^.baseadr,true,true);
  END;
(***************************************)
procedure delayer(x:word);
var i : word;
    a : byte;
begin
  for i:=1 to x do
    begin
      a:=port[$21];  (* I/O access is independent of PC clock *)
    end;
end;
(***************************************)
function in_iic(pointr:IK410_pointr):boolean;
var
  buffer : word;
begin
  scl0_sda1(pointr);
  delayer(10);
  scl1_sda1(pointr);
  delayer(10);
  in_iic:=read_iic(pointr^.baseadr);
  scl1_sda1(pointr);
  delayer(10);
  scl0_sda1(pointr);
  delayer(10);
end;
(***************************************)
procedure start_iic(pointr:IK410_pointr);
begin
  scl0_sda1(pointr);
  delayer(10);
  scl1_sda1(pointr);
  delayer(10);
  scl1_sda0(pointr);
  delayer(10);
  scl0_sda0(pointr);
  delayer(10);
end;
(***************************************)
procedure stopp_iic(pointr:IK410_pointr);
begin
  scl0_sda0(pointr);
  delayer(10);
  scl1_sda0(pointr);
  delayer(10);
  scl1_sda1(pointr);
  delayer(10);
  scl0_sda1(pointr);
  delayer(5000);  (* V1.01 *)
end;
(***************************************)
procedure out_0_iic(pointr:IK410_pointr);
begin
  scl0_sda0(pointr);
  delayer(10);
  scl1_sda0(pointr);
  delayer(10);
  scl0_sda0(pointr);
  delayer(10);
end;
(***************************************)
procedure out_1_iic(pointr:IK410_pointr);
begin
  scl0_sda1(pointr);
  delayer(10);
  scl1_sda1(pointr);
  delayer(10);
  scl0_sda1(pointr);
  delayer(10);
end;
(**************************************************)
(*   Medium-Level Procedures: Byte Manipulation   *)
(**************************************************)
procedure poll_poti(pointr:IK410_pointr;var error:boolean);
var
  bit           : boolean;
  timeout       : word;
begin
    timeout:=0;
    repeat
        start_iic(pointr);
        out_0_iic(pointr);
        out_1_iic(pointr);
        out_0_iic(pointr);
        out_1_iic(pointr);
        out_0_iic(pointr);   (* A3 = 0 *)
        out_0_iic(pointr);   (* A2 = 0 *)
        out_0_iic(pointr);   (* A1 = 0 *)
        out_0_iic(pointr);   (* A0 = 0 *)
        bit:=in_iic(pointr);
        if bit then stopp_iic(pointr);
        timeout:=timeout+1;
    until (not(bit)) or (timeout=100);
    error:=bit;
end;
(***************************************)
procedure poti_write2(pointr:IK410_pointr;instruct,poti,reg:byte;var error:boolean);
var
  bit           : byte;
begin
    poll_poti(pointr,error);
    if (not error) then
    begin
      bit:=instruct and $08;
      if bit=$08 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $04;
      if bit=$04 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=reg and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=reg and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      error:=error or in_iic(pointr);
    end;
  stopp_iic(pointr);
end;
(***************************************)
procedure poti_write3(pointr:IK410_pointr;instruct,poti,reg,data:byte;var error:boolean);
var
  bit            : byte;
begin
    poll_poti(pointr,error);
    if (not error) then
    begin
      bit:=instruct and $08;
      if bit=$08 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $04;
      if bit=$04 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=reg and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=reg and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      error:=error or in_iic(pointr);
      if (not error) then
      begin
        out_0_iic(pointr);
        out_0_iic(pointr);
        bit:=data and $20;
        if bit=$20 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $10;
        if bit=$10 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $08;
        if bit=$08 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $04;
        if bit=$04 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $02;
        if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $01;
        if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
        error:=error or in_iic(pointr);
      end;
    end;
 stopp_iic(pointr);
end;
(***************************************)
procedure poti_inc(pointr:IK410_pointr;poti,turns:byte;var error:boolean);
var
  bit,i : byte;
begin
    poll_poti(pointr,error);
    if (not error) then
    begin
      out_0_iic(pointr);    (* Increment/Decrement *)
      out_0_iic(pointr);
      out_1_iic(pointr);
      out_0_iic(pointr);
      bit:=poti and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      out_0_iic(pointr);
      out_0_iic(pointr);
      error:=error or in_iic(pointr);
      if (not error) then
      begin
        if turns>8 then turns:=8;
        for i:=1 to turns do
          begin
            out_1_iic(pointr);
          end;
      end;
    end;
 stopp_iic(pointr);
end;
(***************************************)
procedure poti_dec(pointr:IK410_pointr;poti,turns:byte;var error:boolean);
var
  bit,i : byte;
begin
    poll_poti(pointr,error);
    if (not error) then
    begin
      out_0_iic(pointr);    (* Increment/Decrement *)
      out_0_iic(pointr);
      out_1_iic(pointr);
      out_0_iic(pointr);
      bit:=poti and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      out_0_iic(pointr);
      out_0_iic(pointr);
      error:=error or in_iic(pointr);
      if (not error) then
      begin
        if turns>8 then turns:=8;
        for i:=1 to turns do
          begin
            out_0_iic(pointr);
          end;
      end;
    end;
 stopp_iic(pointr);
end;
(***************************************)
function poti_read(pointr:IK410_pointr;instruct,poti,reg:byte;var error:boolean):byte;
var
  bit,data       : byte;
  inbit          : boolean;
begin
    poll_poti(pointr,error);
    if (not error) then
    begin
      bit:=instruct and $08;
      if bit=$08 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $04;
      if bit=$04 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=reg and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=reg and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      error:=error or in_iic(pointr);
      if (not error) then
      begin
        data:=0;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $80;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $40;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $20;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $10;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $08;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $04;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $02;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $01;
        out_1_iic(pointr);      (* No acknowledge *)
      end;
    end;
 stopp_iic(pointr);
 poti_read:=data;
end;
(***************************************)
(*    High-Level Procedures            *)
(***************************************)
function Read_phasepoti(pointr:IK410_pointr;var error:boolean):byte;
var
  value:byte;
begin
   value:=poti_read(pointr,$09,$00,$00,error);
   Read_phasepoti:=value;
end;
(***************************************)
function Read_sympoti(pointr:IK410_pointr;var error:boolean):byte;
var
  value : byte;
begin
 value:=poti_read(pointr,$09,$01,$00,error);
 Read_sympoti:=value;
end;
(***************************************)
procedure Write_phasepoti(pointr:IK410_pointr;value:byte;var error:boolean);
var
  error0,error1 : boolean;
begin
  poti_write3(pointr,$0C,$00,$00,value,error0);
  poti_write2(pointr,$0D,$00,$00,error1);
  error:=error0 or error1;
end;
(***************************************)
procedure Write_sympoti(pointr:IK410_pointr;value:byte;var error:boolean);
var
  error0,error1 : boolean;
begin
  poti_write3(pointr,$0C,$01,$00,value,error0);
  poti_write2(pointr,$0D,$01,$00,error1);
  error:=error0 or error1;
end;
(***************************************)
procedure Write_offset00(pointr:IK410_pointr;value:integer);
begin
  if value<-63 then value:=-63;
  if value>63 then value:=63;
  pointr^.offset00:=value;
  write_offset(pointr^.baseadr,
               pointr^.offset00,
               pointr^.offset90);
end;
(***************************************)
procedure Write_offset90(pointr:IK410_pointr;value:integer);
begin
  if value<-63 then value:=-63;
  if value>63 then value:=63;
  pointr^.offset90:=value;
  write_offset(pointr^.baseadr,
               pointr^.offset00,
               pointr^.offset90);
end;
(***************************************)
procedure Turn_phasepoti(pointr:IK410_pointr;turns:byte;updown:boolean;var error:boolean);
begin
  if updown then poti_inc(pointr,0,turns,error) else poti_dec(pointr,0,turns,error);
end;
(***************************************)
procedure Turn_sympoti(pointr:IK410_pointr;turns:byte;updown:boolean;var error:boolean);
begin
  if updown then poti_inc(pointr,1,turns,error) else poti_dec(pointr,1,turns,error);
end;
(***************************************)
procedure Turn_offsetdg00(pointr:IK410_pointr;turns:byte;updown:boolean);
var i : byte;
begin
  for i:=1 to turns do
    begin
      if updown then
        begin
          if (pointr^.offset00<63) then
            begin
              pointr^.offset00:=pointr^.offset00+1
            end;
        end
      else
        begin
          if (pointr^.offset00>-63) then
            begin
              pointr^.offset00:=pointr^.offset00-1;
            end;
        end;
    end;
   write_offset(pointr^.baseadr,
                pointr^.offset00,
                pointr^.offset90);
end;
(***************************************)
procedure Turn_offsetdg90(pointr:IK410_pointr;turns:byte;updown:boolean);
var i : byte;
begin
  for i:=1 to turns do
    begin
      if updown then
        begin
          if (pointr^.offset90<63) then
            begin
              pointr^.offset90:=pointr^.offset90+1
            end;
        end
      else
        begin
          if (pointr^.offset90>-63) then
            begin
              pointr^.offset90:=pointr^.offset90-1;
            end;
        end;
    end;
  write_offset(pointr^.baseadr,
               pointr^.offset00,
               pointr^.offset90);
end;
(***************************************)
procedure store_potis(pointr:IK410_pointr;var error:boolean);
begin
  (* Global write to all wiper registers in register 0 *)
  poti_write2(pointr,$08,$00,$00,error);
end;
(***************************************)



(*******************************************)
(*   Default Values for all Pot Registers  *)
(*******************************************)
procedure Poti_default(pointr:IK410_pointr;var error:boolean);
var
  err : boolean;
begin
  (* Pot in neutral position *)

  Write_phasepoti(pointr,32,error);
  error:=error or err;
  Write_sympoti(pointr,32,err);
  error:=error or err;

  (* Reset register 1,2,3 of pots 0,1,2,3 *)

  poti_write3(pointr,$0C,$00,$01,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$00,$02,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$00,$03,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$01,$01,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$01,$02,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$01,$03,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$02,$01,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$02,$02,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$02,$03,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$03,$01,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$03,$02,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$03,$03,$00,err);
  error:=error or err;

end;
(***********************************************)
(*   Offset is only stored in EEPROM Registers *)
(***********************************************)
procedure Read_2_byte(pointr:IK410_pointr;var off0_1,off9_1:byte;var error:boolean);
var
  a   : byte;
  err : boolean;
begin
  (* Load offset from register 2 *)
  a:=poti_read(pointr,$0B,$00,$02,error);
  a:=a and $0F;
  off0_1:=a;
  a:=poti_read(pointr,$0B,$00,$03,err);
  error:=error or err;
  a:=a and $0F;
  off0_1:=off0_1+(a shl 4);

  a:=poti_read(pointr,$0B,$01,$02,err);
  error:=error or err;
  a:=a and $0F;
  off9_1:=a;
  a:=poti_read(pointr,$0B,$01,$03,err);
  error:=error or err;
  a:=a and $0F;
  off9_1:=off9_1+(a shl 4);

end;
(***************************************)
procedure Write_2_byte(pointr:IK410_pointr;off0_1,off9_1:byte;var error:boolean);
var
  a   : byte;
  err : boolean;
begin
  (* Offset wird in Register 2 gespeichert *)
  a:=off0_1 and $0F;
  poti_write3(pointr,$0C,$00,$02,a,error);
  a:=off0_1 shr 4;
  a:=a and $0F;
  poti_write3(pointr,$0C,$00,$03,a,err);
  error:=error or err;

  a:=off9_1 and $0F;
  poti_write3(pointr,$0C,$01,$02,a,err);
  error:=error or err;
  a:=off9_1 shr 4;
  a:=a and $0F;
  poti_write3(pointr,$0C,$01,$03,a,err);
  error:=error or err;

end;
(***************************************)
PROCEDURE Load_offset(board:IK410_pointr;var error:boolean);  (* Read EEPROM *)
  VAR
    a,b     : byte;
    w       : integer;
  BEGIN
    Read_2_byte(board,a,b,error);
    if a<128 then
      begin
        w:=a;
      end
    else
      begin
        w:=not(a)+1;
        w:=-w;
      end;
    board^.offset00:=w;
    if b<128 then
      begin
        w:=b;
      end
    else
      begin
        w:=not(b)+1;
        w:=-w;
      end;
    board^.offset90:=w;
  END;
(***************************************)
PROCEDURE Store_offset(board:IK410_pointr;var error:boolean);  (* Write EEPROM *)
  VAR
    a,b : byte;
  BEGIN
    a:=lo(board^.offset00);
    b:=lo(board^.offset90);
    Write_2_byte(board,a,b,error);
  END;
(***************************************)
(*      IIC End                        *)
(***************************************)
