/*----------------------------IK320.C---------------------------------

  DR. JOHANNES HEIDENHAIN GmbH, Traunreut, Germany

  Driver Unit for IK 320

  V 1.00
  September 1995
  ------------------------------------------------------------------*/
#include <stdlib.h>
#include <conio.h>
#include <stdio.h>
#include <dos.h>
#include <process.h>
#include "vmerotec.h"//Header file for ROTEC VME interface
#include "ik320.h"
#include "sample.h"


/*--------------------------------------------------------------------
  Definition of an global union for IK interrupt status word
  ------------------------------------------------------------------*/

static struct TWOBYTES  {unsigned char uc0, uc1;};
static struct ONEWORD   {unsigned short us;};
static union  WORDBYTE  {struct TWOBYTES tb;
                         struct ONEWORD ow;}stawbStatus;

/*--------------------------------------------------------------------
  Global variables
  ------------------------------------------------------------------*/
static unsigned char staucDIP_Switch_II = DIP_SWITCH_SII;
static unsigned char staucAxis1WasRead, staucAxis2WasRead;
static unsigned char staucAxisComWasRead;
static unsigned char staucREF1Crossed, staucREF2Crossed;
static unsigned char staucInterruptFinished;
static double        stadPositionValue1, stadPositionValue2;
static double        stadPositionValueCom;

extern unsigned char extucErrorCode, extucMessage;
/*--------------------------------------------------------------------
  pOriginalInterruptVector

  Specifies a vector for the original interrupt.
  ------------------------------------------------------------------*/
void (interrupt far *pOriginalInterruptVector)();


/*--------------------------------------------------------------------
  NewInterruptRoutine

  This function specifies the interrupt routine for IK interrupt
  ------------------------------------------------------------------*/
void interrupt far NewInterruptRoutine(void)
{
  unsigned short usAddress;
  
  outp(INTC2A1,inp(INTC2A1) | ~INT_MASK);   //Disable IRQ15
  
  //Reset DOS-interrupt line  
  outp(INTC2A0,EOI);
  outp(INTC1A0,EOI);
  
  _enable();               //Enable DOS interrupt
  	
  usAddress = CALCULATE_IK_ADDRESS(staucDIP_Switch_II);

  SWITCH_VME_TO_A24_ADDRESS_SPACE(staucDIP_Switch_II);
  Read_IK_InterruptStatus(staucDIP_Switch_II);
  Evaluate_IK_InterruptStatus(staucDIP_Switch_II);
  
  //ROTEC specific code: Reset VME interrupt line
  int_eoi(IACK3);
  
  outp(INTC2A1,inp(INTC2A1) & INT_MASK);  //enable IRQ15
                                         
  usAddress = CALCULATE_IK_ADDRESS(staucDIP_Switch_II);
  outport (usAddress + INTSTAT, 0x0000);

  staucInterruptFinished = 1;                       //Interrupt finished
}//End NewInterruptRoutine

/*--------------------------------------------------------------------
  RestoreOldInterruptVector

  This function restores the old interrupt vector.
  ------------------------------------------------------------------*/
void RestoreOldInterruptVector(void)
{
  //Disable hardware interrupts
  _disable();
  //Restore original interrupt vector
  _dos_setvect(INT_NR,pOriginalInterruptVector);
  //Enable hardware interrupts
  _enable();
}//End RestoreOldInterruptVector


/*--------------------------------------------------------------------
  Read_IK_InterruptStatus
  
  This function reads the interrupt status word of the IK.
  ------------------------------------------------------------------*/
void Read_IK_InterruptStatus(unsigned char ucDIP_Switch_II)
{
  short usAddress;
  //Calculate address
  usAddress = CALCULATE_IK_ADDRESS(ucDIP_Switch_II);
  //Read status
  stawbStatus.ow.us = inport (usAddress + INTSTAT);
}//End Read_IK_InterruptStatus

/*--------------------------------------------------------------------
  Evaluate_IK_InterruptStatus

  When the IK sends an interrupt to the master, the cause of the
  interrupt is shown in the interrupt status word. This function
  evaluates the interrupt status word.
  ------------------------------------------------------------------*/
void Evaluate_IK_InterruptStatus(unsigned char ucDIP_Switch_II)
{

  unsigned short usDummy, usAddress;
  long lDummy;

  usAddress = CALCULATE_IK_ADDRESS(ucDIP_Switch_II);

  switch (stawbStatus.tb.uc1)
    {
   case 0x00:
    break;

   case 0x01:
    usDummy=inport (usAddress + POS_X1_1);
    lDummy=(long)(usDummy)<<16;
    usDummy=inport (usAddress + POS_X1_2);
    lDummy+=usDummy;
    stadPositionValue1=(double)lDummy;
    usDummy=inport (usAddress + POS_X1_3);
    stadPositionValue1+=(double)usDummy/(SUBDIVISION*16.);
    outportb (usAddress + TM_X1, (char)0x00);
	  usDummy=inportb  (usAddress + STAT_X1);
    staucAxis1WasRead = 1;
    break;

   case 0x02:
    usDummy=inport (usAddress + POS_X2_1);
    lDummy=(long)(usDummy)<<16;
    usDummy=inport (usAddress + POS_X2_2);
    lDummy+=usDummy;
    stadPositionValue2=(double)lDummy;
    usDummy=inport (usAddress + POS_X2_3);
    stadPositionValue2+=(double)usDummy/(SUBDIVISION*16.);
    outportb (usAddress + TM_X2, (char)0x00);
    usDummy=inportb  (usAddress + STAT_X2);
    staucAxis2WasRead = 1;
    break;

   case 0x03:
    usDummy=inport (usAddress + POS_COMB_1);
    lDummy=(long)(usDummy)<<16;
    usDummy=inport (usAddress + POS_COMB_2);
    lDummy+=usDummy;
    stadPositionValueCom=(double)lDummy;
    usDummy=inport (usAddress + POS_COMB_3);
    stadPositionValueCom+=(double)usDummy/(SUBDIVISION*16.);
    outportb (usAddress + TM_COMB, (char)0x00);
    usDummy=inportb  (usAddress + STAT_COMB);
    staucAxisComWasRead = 1;
    break;

   case 0x04:
    if (stawbStatus.tb.uc0 & 0x01)
     {
     extucErrorCode = 0x01;
     }
    if (stawbStatus.tb.uc0 & 0x02)
     {
     extucErrorCode = 0x02;
     }
    break;

   case 0x05:
    if (stawbStatus.tb.uc0 & 0x01)
     {
     extucErrorCode = 0x03;
     }
    if (stawbStatus.tb.uc0 & 0x02)
     {
     extucErrorCode = 0x04;
     }
    break;

   case 0x06:
    if (stawbStatus.tb.uc0 == 1)
     outportb (usAddress + TM_X1, (char)0x00);//Reset TM_X1
    else if (stawbStatus.tb.uc0 == 2)
     outportb (usAddress + TM_X2, (char)0x00);//Reset TM_X2
    else
	  outportb (usAddress + TM_COMB, (char)0x00);//Reset TM_COMB
    break;

   case 0x07:
    extucMessage = 0x01;
    if (stawbStatus.tb.uc0 & 0x01)
     {
     extucErrorCode = 0x05;
     }
    if (stawbStatus.tb.uc0 & 0x02)
     {
     extucErrorCode = 0x06;
     }
    if (stawbStatus.tb.uc0 & 0x04)
     {
     extucErrorCode = 0x07;
     }
    if (stawbStatus.tb.uc0 & 0x08)
     {
     extucErrorCode = 0x08;
     }
    if (stawbStatus.tb.uc0 & 0x10)
     {
     extucErrorCode = 0x09;
     }
    if (stawbStatus.tb.uc0 & 0x20)
     {
     extucErrorCode = 0x10;
     }
    if (stawbStatus.tb.uc0 & 0x40)
     {
     extucErrorCode = 0x11;
     }
    break;

   case 0x08:
    extucMessage = 0x02;
    staucREF1Crossed = 1;
    break;

   case 0x09:
    extucMessage = 0x03;
    staucREF2Crossed = 1;
    break;

   case 0x0A:
	 if (stawbStatus.tb.uc0 == 0x01)
     {
     extucErrorCode = 0x12;
     }
    break;

   case 0x0B:
    switch (stawbStatus.tb.uc0)
     {
     case 0x00:
      extucMessage = 0x04;
      break;
     case 0x01:
      extucErrorCode = 0x13;
      break;
     case 0x02:
      extucErrorCode = 0x14;
      break;
     case 0x03:
      extucErrorCode = 0x15;
      break;
     case 0x04:
      extucErrorCode = 0x16;
      break;
     case 0x05:
      extucErrorCode = 0x17;
      break;
     case 0x06:
      extucErrorCode = 0x18;
      break;
      case 0x10:
      extucMessage = 0x05;
      break;
     default:
       extucErrorCode = 0x99;
     }
    break;

   case 0x0C:
    switch (stawbStatus.tb.uc0)
     {
     case 0x00:
      extucMessage = 0x06;
      break;
     case 0x01:
      extucErrorCode = 0x19;
      break;
     case 0x02:
      extucErrorCode = 0x20;
      break;
     case 0x03:
      extucErrorCode = 0x21;
      break;
     case 0x04:
      extucErrorCode = 0x22;
      break;
     case 0x05:
      extucErrorCode = 0x23;
      break;
     case 0x06:
      extucErrorCode = 0x24;
      break;
     case 0x10:
      extucMessage = 0x07;
      break;
     default:
      extucErrorCode = 0x99;
     }
    break;

   case 0x0E:
    if (stawbStatus.tb.uc0 == 0x01)
    {
    extucErrorCode = 0x25;
    }
    if (stawbStatus.tb.uc0 == 0x02)
    {
    extucErrorCode = 0x26;
    }
    break;

   case 0x0F:
    if (stawbStatus.tb.uc0 == 0x01)
     {
     extucErrorCode = 0x27;
     }
    if (stawbStatus.tb.uc0 == 0x02)
     {
     extucErrorCode = 0x28;
     }
    if (stawbStatus.tb.uc0 == 0x03)
     {
     extucErrorCode = 0x29;
     }
    break;

   case 0x10:
    extucMessage = 0x08;
    break;

   case 0x11:
    extucMessage = 0x09;
    break;

   case 0xFD:
    switch (stawbStatus.tb.uc0)
     {
	  case 0x01:
      extucErrorCode = 0x30;
      break;
     case 0x02:
      extucErrorCode = 0x31;
      break;
     default:
      extucErrorCode = 0x99;
     }

   case 0xFE:
    extucErrorCode = 0x32;
    break;

   case 0xFF:
    extucErrorCode = 0x33;
    break;

   default:
    extucErrorCode = 0x99;
   }//End switch - stawbStatus.tb.uc1
}//End Evaluate_IK_InterruptStatus

/*--------------------------------------------------------------------
  MasterInterrupt

  This function sets parameter P81 and generates a master interrupt
  ------------------------------------------------------------------*/
void MasterInterrupt(unsigned char ucDIP_Switch_I,
                     unsigned char ucDIP_Switch_II,
                     unsigned short usFunction)
{
  short usAddress, sBasAdrGroup;
  //calculate synchronous latch interrupt address
  sBasAdrGroup = CALCULATE_BAS_ADR_GROUP(ucDIP_Switch_I);
  usAddress = CALCULATE_IK_ADDRESS(ucDIP_Switch_II);

  outport (usAddress + PAR_81,usFunction);

  _disable();
  SWITCH_VME_TO_A16_ADDRESS_SPACE(ucDIP_Switch_I);
  //Execute master interrupt
  outportb ((short)(sBasAdrGroup + ((ucDIP_Switch_I & 0x1F) * 2)),
           (char)0x00);
  SWITCH_VME_TO_A24_ADDRESS_SPACE(ucDIP_Switch_II);
  _enable();

}//End MasterInterrupt

/*--------------------------------------------------------------------
  InitIk320

  This function initializes the IK.
  ------------------------------------------------------------------*/
void InitIK320 (unsigned char ucDIP_Switch_I,
                unsigned char ucDIP_Switch_II)
{
  printf("\nInitialize IK 320  %02x\n",ucDIP_Switch_II);

  //Disable hardware interrupts
  _disable();
  
  //Save old interrupt vector
  pOriginalInterruptVector = _dos_getvect(INT_NR);
  
  //Set new interrupt vector
  _dos_setvect(INT_NR,NewInterruptRoutine);
  
  //Set interrupt controller mask
  outp(INTC2A1,inp(INTC2A1) & INT_MASK);
  outp(INTC1A1,inp(INTC1A1) & ~0x04);
  
  //Set interrupt controller to End of Interrupt
  outp(INTC2A0,EOI);
  outp(INTC1A0,EOI);

  //Enable hardware interrupts
  _enable();

  SWITCH_VME_TO_A24_ADDRESS_SPACE(ucDIP_Switch_II);
                                                             
  InitParams (ucDIP_Switch_II);//Set the parameters
  staucInterruptFinished = 0;
  MasterInterrupt(ucDIP_Switch_I, ucDIP_Switch_II, 0x0007);
  do
  {
   if (extucErrorCode)
    {
    DisplayError();
    return;
    }
   if (extucMessage)
    {
    DisplayMessage();
    }
  }
  while (staucInterruptFinished == 0);//Wait for interrupt
}//End InitIk320


/*--------------------------------------------------------------------
  InitParams

  This function initializes the Parameters.
  ------------------------------------------------------------------*/
void InitParams (unsigned char ucDIP_Switch_II)
{
  SetParam (ucDIP_Switch_II, PAR_01_1, 0x00,0);//Count. direct. axis 1
  SetParam (ucDIP_Switch_II, PAR_01_2, 0x00,0);//Count. direct. axis 2
  SetParam (ucDIP_Switch_II, PAR_01_3, 0x00,0);//Counting direction
                                               //axis combination

  SetParam (ucDIP_Switch_II, PAR_02_1, 0x01,0);//Definition axis 1
  SetParam (ucDIP_Switch_II, PAR_02_2, 0x01,0);//Definition axis 2
  SetParam (ucDIP_Switch_II, PAR_02_3, 0x01,0);//Definition axis comb.

  //Number of bits for subdivision
  SetParam (ucDIP_Switch_II, PAR_03  , INTERPOLATION_BITS,0);

  SetParam (ucDIP_Switch_II, PAR_04_1, 0x0000,0);//Ref. mark spacing 1
  SetParam (ucDIP_Switch_II, PAR_04_2, 0x0000,0);//Ref. mark spacing 2

  SetParam (ucDIP_Switch_II, PAR_05_1, 0x00008CA0,0);//Signal per. 1
  SetParam (ucDIP_Switch_II, PAR_05_2, 0x00000800,0);//Signal per. 2
  SetParam (ucDIP_Switch_II, PAR_05_3, 0x00000000,0);//Signal periods
                                                    //axis combination

  SetParam (ucDIP_Switch_II, PAR_06_1, 0x00,0);//Compensation on/off 1
  SetParam (ucDIP_Switch_II, PAR_06_2, 0x00,0);//Compensation on/off 2

  SetParam (ucDIP_Switch_II, PAR_07_1, 0x00000000,0);//Comp. start 1
  SetParam (ucDIP_Switch_II, PAR_07_2, 0x00000000,0);//Comp. start 2

  SetParam (ucDIP_Switch_II, PAR_08_1, 0x0100,0);//Nr. of comp. pts. 1
  SetParam (ucDIP_Switch_II, PAR_08_2, 0x0100,0);//Nr. of comp. pts. 2

  SetParam (ucDIP_Switch_II, PAR_09_1, 0x0010,0);//Interv. of c. pts 1
  SetParam (ucDIP_Switch_II, PAR_09_2, 0x0010,0);//Interv. of c. pts 2

  SetParam (ucDIP_Switch_II, PAR_10  , 0x00,0);//Latch enable

  SetParam (ucDIP_Switch_II, PAR_19_1, 0x00000000,0);//Ref. offs. 1
  SetParam (ucDIP_Switch_II, PAR_19_2, 0x00000000,0);//Ref. offs. 2                  

  SetParam (ucDIP_Switch_II, PAR_21  , 0x00,0);//Axis combination

														//Compensation run
  SetParam (ucDIP_Switch_II, PAR_30_1, 0x01,0);//Dir.(axis 1) &
														  //freq.(axis 1+2)
  SetParam (ucDIP_Switch_II, PAR_30_2, 0x00,0);//Direction axis 2

  SetParam (ucDIP_Switch_II, PAR_70_1, 0x00000000,0);//Ext. preset 1
  SetParam (ucDIP_Switch_II, PAR_70_2, 0x00000000,0);//Ext. preset 2
  SetParam (ucDIP_Switch_II, PAR_70_3, 0x00000000,0);//Ext. preset
																//axis combination

  SetParam (ucDIP_Switch_II, PAR_71_1, 0x00000000,0);//Bus preset 1
  SetParam (ucDIP_Switch_II, PAR_71_2, 0x00000000,0);//Bus preset 2
  SetParam (ucDIP_Switch_II, PAR_71_3, 0x00000000,0);//Bus preset
																  //axis combination

  SetParam (ucDIP_Switch_II, PAR_72_1, 0x00000000,0);//Axis offset 1
  SetParam (ucDIP_Switch_II, PAR_72_2, 0x00000000,0);//Axis offset 2
  SetParam (ucDIP_Switch_II, PAR_72_3, 0x00000000,0);//Axis offset
																  //axis combination

  SetParam (ucDIP_Switch_II, PAR_80_1, 0x00,0);//External function 1
  SetParam (ucDIP_Switch_II, PAR_80_2, 0x00,0);//External function 2

}//End InitParams


/*--------------------------------------------------------------------
  SetParam

  With this function the master sets the parameters.
  ------------------------------------------------------------------*/
void SetParam (unsigned char ucDIP_Switch_II,
					unsigned short usOffsetAddress, long lData,short sData)
{
  unsigned short usBaseAddress;
  usBaseAddress = CALCULATE_IK_ADDRESS(ucDIP_Switch_II);
  switch (usOffsetAddress)
	{
  case PAR_01_1:
	outportb (usBaseAddress+usOffsetAddress, (char) lData);
	break;
  case PAR_01_2:
	outportb (usBaseAddress+usOffsetAddress, (char) lData);
	break;
  case PAR_01_3:
	outportb (usBaseAddress+usOffsetAddress, (char) lData);
	break;
  case PAR_02_1:
	outportb (usBaseAddress+usOffsetAddress, (char) lData);
	break;
  case PAR_02_2:
	outportb (usBaseAddress+usOffsetAddress, (char) lData);
	break;
  case PAR_02_3:
	outportb (usBaseAddress+usOffsetAddress, (char) lData);
	break;
  case PAR_03  :
	outport (usBaseAddress+usOffsetAddress, (short)  lData);
	break;
  case PAR_04_1:
	outport (usBaseAddress+usOffsetAddress, (short)  lData);
	break;
  case PAR_04_2:
	outport (usBaseAddress+usOffsetAddress, (short)  lData);
	break;
  case PAR_05_1:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	break;
  case PAR_05_2:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	break;
  case PAR_05_3:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	break;
  case PAR_06_1:
	outportb (usBaseAddress+usOffsetAddress, (char) lData);
	break;
  case PAR_06_2:
	outportb (usBaseAddress+usOffsetAddress, (char) lData);
	break;
  case PAR_07_1:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	break;
  case PAR_07_2:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	break;
  case PAR_08_1:
	outport (usBaseAddress+usOffsetAddress, (short)  lData);
	break;
  case PAR_08_2:
	outport (usBaseAddress+usOffsetAddress, (short)  lData);
	break;
  case PAR_09_1:
	outport (usBaseAddress+usOffsetAddress, (short)  lData);
	break;
  case PAR_09_2:
	outport (usBaseAddress+usOffsetAddress, (short)  lData);
	break;
  case PAR_10  :
	outportb (usBaseAddress+usOffsetAddress, (char) lData);
	break;
  case PAR_19_1:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	break;
  case PAR_19_2:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	break;
  case PAR_21  :
	outportb (usBaseAddress+usOffsetAddress, (char) lData);
	break;
  case PAR_30_1:
	outportb (usBaseAddress+usOffsetAddress, (char) lData);
	break;
  case PAR_30_2:
	outportb (usBaseAddress+usOffsetAddress, (char) lData);
	break;
  case PAR_70_1:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	outport (usBaseAddress+usOffsetAddress+4, sData);
	break;
  case PAR_70_2:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	outport (usBaseAddress+usOffsetAddress+4, sData);
	break;
  case PAR_70_3:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	outport (usBaseAddress+usOffsetAddress+4, sData);
	break;
  case PAR_71_1:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	outport (usBaseAddress+usOffsetAddress+4, sData);
	break;
  case PAR_71_2:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	outport (usBaseAddress+usOffsetAddress+4, sData);
	break;
  case PAR_71_3:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	outport (usBaseAddress+usOffsetAddress+4, sData);
	break;
  case PAR_72_1:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	outport (usBaseAddress+usOffsetAddress+4, sData);
	break;
  case PAR_72_2:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	outport (usBaseAddress+usOffsetAddress+4, sData);
	break;
  case PAR_72_3:
	outport (usBaseAddress+usOffsetAddress,   (short) (lData >> 16));
	outport (usBaseAddress+usOffsetAddress+2, (short) (lData & 0xffff));
	outport (usBaseAddress+usOffsetAddress+4, sData);
	break;
  case PAR_80_1:
	outportb (usBaseAddress+usOffsetAddress, (char) lData);
	break;
  case PAR_80_2:
	outportb (usBaseAddress+usOffsetAddress, (char) lData);
	break;
  default:
   gotoxy(1,23);
   puts ("Error: Wrong parameter number");
	 }//End switch (usOffsetAddress)
}//End SetParam



/*--------------------------------------------------------------------
  TraverseOverReferenceMark

  This function causes the operator to traverse over the reference
  mark.
  ------------------------------------------------------------------*/
void TraverseOverReferenceMark(unsigned char ucDIP_Switch_I,
										 unsigned char ucDIP_Switch_II,
                               unsigned char ucAxis)
{
  clrscr();
  switch (ucAxis)
  {
  case 1:
   printf("\nCross over reference mark of axis 1\n");
   staucREF1Crossed = 0;
   MasterInterrupt(ucDIP_Switch_I, ucDIP_Switch_II, 0x0008);

   do
    {
    if (extucErrorCode)
     {
     DisplayError();
     break;
     }
    if (extucMessage)
     {
     DisplayMessage();
     }
    }
   while (!(staucREF1Crossed));
   break;
  case 2:
   printf("Cross over reference mark of axis 2\n");
   staucREF2Crossed = 0;
   MasterInterrupt(ucDIP_Switch_I, ucDIP_Switch_II, 0x0009);
   do
    {
     if (extucErrorCode)
      {
      DisplayError();
      break;
      }
     if (extucMessage)
      {
      DisplayMessage();
      }
     }
     while (!(staucREF2Crossed));
   break;
  default:
   gotoxy (1,23);
   puts ("Error: Wrong axis number");
  }
}//End TraverseOverReferenceMark

/*--------------------------------------------------------------------
  SynchroPosTrigger

  This function triggers axis 1 and axis 2 synchronously
  ------------------------------------------------------------------*/
void SynchroPosTrigger(unsigned char ucDIP_Switch_I,
						  unsigned char ucDIP_Switch_II)
{
  short sBasAdrGroup;
  //calculate synchronous latch interrupt address
  sBasAdrGroup = CALCULATE_BAS_ADR_GROUP(ucDIP_Switch_I);
  
  _disable();
  SWITCH_VME_TO_A16_ADDRESS_SPACE(ucDIP_Switch_I);
  outportb (sBasAdrGroup, (char)0x00);//Synchronous latch
  SWITCH_VME_TO_A24_ADDRESS_SPACE(ucDIP_Switch_II);
  _enable();

}//End SynchroPosTrigger

/*--------------------------------------------------------------------
  DisplayPositionValue

  This function displays the actual position
  ------------------------------------------------------------------*/
void DisplayPositionValue(unsigned char ucDIP_Switch_I,
								  unsigned char ucDIP_Switch_II)
{
  clrscr();
  printf("\n\n");
  do
  {
  SynchroPosTrigger(ucDIP_Switch_I, ucDIP_Switch_II);

   do
    {
    if (extucErrorCode)
     {
     DisplayError();
     return;
     }
    }
   while (!(staucAxis1WasRead && staucAxis2WasRead));

   printf("\r\t%10.4f\t%10.4f",stadPositionValue1,stadPositionValue2);
   staucAxis1WasRead = 0;
   staucAxis2WasRead = 0;
  }
  while (!kbhit());getch();
}//End DisplayPositionValue

/*--------------------------------------------------------------------
  CompensationRun

  This function starts the compensation run
  ------------------------------------------------------------------*/
void CompensationRun(unsigned char ucDIP_Switch_I,
							unsigned char ucDIP_Switch_II,
							unsigned char ucAxis,
							unsigned char ucCompensationStart,
							unsigned short usNumberOfCompPoints,
							unsigned short usIntervalOfCompPoints,
							short sDirectionAndFrequency)
{
  switch(ucAxis)
	{
	case 1: SetParam (ucDIP_Switch_II, PAR_06_1, 0x01,0);//Comp. on/off
	 SetParam (ucDIP_Switch_II, PAR_07_1, ucCompensationStart,0);
	 SetParam (ucDIP_Switch_II, PAR_08_1, usNumberOfCompPoints,0);
	 SetParam (ucDIP_Switch_II, PAR_09_1, usIntervalOfCompPoints,0);
	 SetParam (ucDIP_Switch_II, PAR_30_1, sDirectionAndFrequency,0);
	 break;
	case 2: SetParam (ucDIP_Switch_II, PAR_06_2, 0x01,0);//Comp. on/off
	 SetParam (ucDIP_Switch_II, PAR_07_2, ucCompensationStart,0);
	 SetParam (ucDIP_Switch_II, PAR_08_2, usNumberOfCompPoints,0);
	 SetParam (ucDIP_Switch_II, PAR_09_2, usIntervalOfCompPoints,0);
	 SetParam (ucDIP_Switch_II, PAR_30_2, sDirectionAndFrequency,0);
	 break;
	default:
    gotoxy (1,23);
	 puts ("Error: The input value for axis must be 1 or 2\n");
    return;
	}

   staucInterruptFinished = 0;
   //Update parameter
   MasterInterrupt(ucDIP_Switch_I, ucDIP_Switch_II, 0x000a);

   do
   if (extucErrorCode)
    {
    DisplayError();
    return;
    }
    while (staucInterruptFinished == 0);//Wait for Interrupt
    
   clrscr();
	 printf ("\nRecord compensation values axis %2d \n\n\
             - press any key to start\n\n\n\n", ucAxis);
	 do
	  {
	  SynchroPosTrigger(ucDIP_Switch_I, ucDIP_Switch_II);

	  do
     {
     if (extucErrorCode)
      {
      DisplayError();
      return;
      }
     }
    while (!(staucAxis1WasRead && staucAxis2WasRead));

    printf("\r\t%10.4f\t%10.4f",stadPositionValue1,stadPositionValue2);
    staucAxis1WasRead =0;
    staucAxis2WasRead =0;
    }
	 while (!kbhit());getch();
   clrscr();

   switch(ucAxis)
	  {
    //Compensation run axis 1
   case 1:
    MasterInterrupt(ucDIP_Switch_I, ucDIP_Switch_II, 0x000b);
    printf ("\nRecord of compensation points for axis 1 started\n\n");
    printf ("- Start axis\n");
	  break;
   //Compensation run axis 2
   case 2:
    MasterInterrupt(ucDIP_Switch_I, ucDIP_Switch_II, 0x000c);
    printf ("\nRecord of compensation points for axis 2 started\n\n");
    printf ("- Start axis\n");
	  break;
	 }
  do
   {
   if (extucErrorCode)
    {
    DisplayError();
    return;
    }
   }
  while (!(extucMessage));
  clrscr();
  DisplayMessage();

  do
   {
   if (extucErrorCode)
    {
    DisplayError();
    return;
    }
   }
  while (!(extucMessage));
  clrscr();
  DisplayMessage();

  do
   {
   if (extucErrorCode)
    {
    DisplayError();
    return;
    }
   }
  while (!(kbhit()));
  getch();
}//End CompensationRun

/*-------------------------------------------------------------------
  CompensationOnOff

  This function switches the signal compensation on or off.
  ------------------------------------------------------------------*/
void CompensationOnOff(unsigned char ucDIP_Switch_I,
                       unsigned char ucDIP_Switch_II)
{
  char cCharacter,cAxis;
  fflush(stdin);
  printf("\n Axis 1 or 2?   ");
  do
   {
   if (extucErrorCode)
    {
    DisplayError();
    return;
    }
   }
  while (!kbhit());

  cAxis=getche();

  fflush(stdin);
  printf("\n Compensation on = c / off = o   ");
  do
   {
   if (extucErrorCode)
    {
    DisplayError();
    return;
    }
   }
  while (!kbhit());

  cCharacter=getche();

  switch(cAxis)
   {
   //Switch compensation on/off axis 1
   case '1':
    if (cCharacter=='c')
     SetParam (ucDIP_Switch_II, PAR_06_1, 0x01,0);
    if (cCharacter=='o')
     SetParam (ucDIP_Switch_II, PAR_06_1, 0x00,0);
    break;
   //Switch compensation on/off axis 2
   case '2':
    if (cCharacter=='c')
    SetParam (ucDIP_Switch_II, PAR_06_2, 0x01,0);
    if (cCharacter=='o')
    SetParam (ucDIP_Switch_II, PAR_06_2, 0x00,0);
    break;
   default:
    gotoxy (1,23);
    puts ("Error: Wrong character");
   }
   //Update parameter
   MasterInterrupt(ucDIP_Switch_I, ucDIP_Switch_II, 0x000a);
}//End CompensationOnOff

/*-------------------------------------------------------------------
  DisplayMessage

  This function displays the messages of the IK interrupt status.
  ------------------------------------------------------------------*/

void DisplayMessage(void)
{
  gotoxy (1,20);
  switch (extucMessage)
   {
   case 0x01:
    puts ("POST concluded\n");
    break;
   case 0x02:
    puts ("REF axis 1 was crossed over\n");
    break;
   case 0x03:
    puts ("REF axis 2 was crossed over\n");
    break;
   case 0x04:
    puts("Compensation run ended axis 1 - Press any key\n");
    break;
   case 0x05:
    puts ("Record of Compensation Points ended axis 1\
           - Please wait\n");
    break;
   case 0x06:
    puts ("Compensation run ended axis 2 - Press any key\n");
    break;
   case 0x07:
    puts ("Record of Compensation Points ended axis 2\
           - Please wait\n");
    break;
   case 0x08:
    puts ("Preset set via master\n");
    break;
   case 0x09:
    puts ("Preset set via external function\n");
    break;
   default:
    puts ("Unknown message code");
   }
  extucMessage = 0;
  delay (3000);
}//End DisplayMessage

/*-------------------------------------------------------------------
  DisplayError

  This function displays the error messages of the
  IK interrupt status.
  ------------------------------------------------------------------*/

void DisplayError(void)
{
  gotoxy (1,23);
  switch (extucErrorCode)
   {
   case 0x01:
    puts ("Error: No latch on axis 1\n");
    break;
   case 0x02:
    puts ("Error: No latch on axis 2\n");
    break;
   case 0x03:
    puts ("Error: Double latch on axis 1\n");
    break;
   case 0x04:
    puts ("Error: Double latch on axis 2\n");
    break;
   case 0x05:
    puts ("Error: CRC correction value 1\n");
    break;
   case 0x06:
    puts ("Error: CRC correction value 2\n");
    break;
   case 0x07:
    puts("Error: parameter setting\n");
    break;
   case 0x08:
    puts("Error: CRC Eprom\n");
    break;
   case 0x09:
    puts ("Error: checksum EPROM 1\n");
    break;
   case 0x10:
    puts ("Error: checksum EPROM 2\n");
    break;
   case 0x11:
    puts ("Error: hardware\n");
    break;
   case 0x12:
    puts ("Error: wrong parameter setting\n");
    break;
   case 0x13:
    puts ("Error: axis 1 has no absolute reference\n");
    break;
   case 0x14:
    puts ("Error: axis 1 has wrong speed\n");
    break;
   case 0x15:
    puts ("Error: axis 1 has wrong position\n");
    break;
   case 0x16:
    puts ("Error: axis 1 has wrong direction\n");
    break;
   case 0x17:
    puts ("Error: axis 1 has wrong number of measuring\
           interrupts\n");
    break;
   case 0x18:
    puts ("Error: wrong calculation during compensation run\
           of axis 1\n");
    break;
   case 0x19:
    puts ("Error: axis 2 has no absolute reference\n");
    break;
   case 0x20:
    puts ("Error: axis 2 has wrong speed\n");
    break;
   case 0x21:
    puts ("Error: axis 2 has wrong position\n");
    break;
   case 0x22:
    puts ("Error: axis 2 has wrong direction\n");
    break;
   case 0x23:
    puts ("Error: axis 2 has wrong number of measuring\
           interrupts\n");
    break;
   case 0x24:
    puts ("Error: wrong calculation during compensation run\
           of axis 2\n");
    break;
   case 0x25:
    puts ("Error: REF distance: axis 1\n");
    break;
   case 0x26:
    puts ("Error: REF distance: axis 2\n");
    break;
   case 0x27:
    puts("Error: CRC - EPROM\n");
    break;
   case 0x28:
    puts ("Error: CRC - Compensation values axis 1\n");
    break;
   case 0x29:
    puts ("Error: CRC - Compensation values axis 2\n");
    break;
   case 0x30:
    puts ("Error: Stack of main program incorrect\n");
    break;
   case 0x31:
    puts ("Error: Memory area for stack is exceeded\n");
    break;
   case 0x32:
    puts ("Error:Hardware defective\n");
    break;
   case 0x33:
    puts ("Error:IK has received Unknown command\n");
    break;
   case 0x99:
    puts ("Error: Unknown code for interrupt status");
    break;
   default:
    puts ("Unknown error code");
   }
  extucErrorCode = 0;
  delay (3000);
}//End DisplayError
