UNIT IK121_0;

{-------------------------------------------------------
  DR. JOHANNES HEIDENHAIN GmbH, Traunreut, Germany

  Driver Unit for IK121 (Basic Functions)

  V 1.00
  July 1994

  V 1.04
  Mai 1998 (dir in read_count_value32/48)
 -----------------------------------------------------------}

{$N+,E+}
{$V+}
{$R+}

interface

PROCEDURE write_g26(baseadr:word;axes,address:byte;datum:word);
FUNCTION read_g26(baseadr:word;axes,address:byte):word;
PROCEDURE soft_l0(baseadr:word;axis:byte);
PROCEDURE soft_l1(baseadr:word;axis:byte);
FUNCTION read_count_value48(baseadr:word;axis,reg:byte;dir:boolean):comp;
FUNCTION read_count_value32(baseadr:word;axis,reg:byte;dir:boolean):comp;
FUNCTION latched(baseadr:word;axis,reg:byte):boolean;
PROCEDURE poll_latch(baseadr:word;axis,reg:byte);

Implementation

{-------------------------------------------------------
                      write_g26
 -------------------------------------------------------
 This procedure writes a value in a 16-bit
 register of a counter.
 -----------------------Parameters----------------------
 baseadr   basic address A0 to A9 of the IK 121
 axis	   axis select - axis 1 or axis 2
 address   address of the counter B0 to B4
 datum	   value to write to the address
 adr_reg   address of the address register on the IK 121
 adr_point the part of the address indicated by the
           address register (R0 bis R2)
 adr_gate  port address in which <datum> is written
 -------------------------------------------------------}

PROCEDURE
 write_g26(baseadr:word;axes,address:byte;datum:word);
  VAR adr_reg,adr_point,adr_gate : word;

  BEGIN
       (* Discard the last four bits of the card
       address *)
    baseadr:=baseadr and $0FF0;
       (* Address B0 to B4 of the counter *)
    address:=address and $001F;

    (* Load address pointer into address register *)
       (* Adress of the address register *)
    adr_reg:=baseadr or $0008;
       (* Value of the address register R0 to R2 =
       address of the counter without B0 and B1 *)
    adr_point:=address shr 2;
       (* Load address register *)
    portw[adr_reg]:=adr_point;

    (* Calculate port address *)
    if axes=1 then
      adr_gate:=baseadr or (address and $03)
    else
      adr_gate:=(baseadr or $0004) or (address and $03);

       (* Write <datum> to the counter *)
    portw[adr_gate]:=datum;
  END;

{-------------------------------------------------------
                        read_g26
 -------------------------------------------------------
 This function reads a value from a 16-bit
 register of a counter.
 --------------------Parameters-------------------------
 baseadr   basic address A0 to A9 of the IK 121
 axis      axis select - axis 1 or axis 2
 address   address of the counter B0 to B4
 adr_reg   address of the address register on the IK 121
 adr_point the part of the address indicated by the
           address register (R0 bis R2)
 adr_gate  port address in which <datum> is written
 -------------------------------------------------------}

FUNCTION read_g26(baseadr:word;axes,address:byte):word;
  VAR adr_reg,adr_point,adr_gate : word;

  BEGIN
       (* Discard the last four bits of the card
       address *)
    baseadr:=baseadr and $0FF0;
       (* Address B0 to B4 of the counter *)
    address:=address and $001F;

    (* Load address pointer into address register *)
       (* Address of the address register *)
    adr_reg:=baseadr or $0008;
       (* Value of the address register R0 to R2 =
       address of the counter without B0 and B1 *)
    adr_point:=address shr 2;
       (* Load address register *)
    portw[adr_reg]:=adr_point;

    (* Calculate port address *)
    if axes=1 then
      adr_gate:=baseadr or (address and $03)
    else
      adr_gate:=(baseadr or $0004) or (address and $03);

       (* Read datum from the counter *)
    read_g26:=portw[adr_gate];
  END;

{-------------------------------------------------------
                         soft_l0
 -------------------------------------------------------
 This procedure reads the measured value and stores
 it in data register 0.
 -------------------------------------------------------}

PROCEDURE soft_l0(baseadr:word;axis:byte);
  BEGIN
    write_g26(baseadr,axis,14,$0001);
  END;

{-------------------------------------------------------
                         soft_l1
 -------------------------------------------------------
 This procedure reads the measured value and stores
 it in data register 1.
 -------------------------------------------------------}

PROCEDURE soft_l1(baseadr:word;axis:byte);
  BEGIN
    write_g26(baseadr,axis,14,$0002);
  END;

{-------------------------------------------------------
                    read_count_value48
 -------------------------------------------------------
 This function reads 48 bits of a measured value.
 -------------------------------------------------------}

FUNCTION
 read_count_value48(baseadr:word;axis,reg:byte;dir:boolean):comp;
  TYPE
    vartype = (li,by);
    mapper = record
               case wert:vartype of
                 li : (field0:comp);
                 by : (field1:array[0..3] of word);
               end;
  VAR
    buffer : mapper;
  BEGIN
    case reg of
     0 : begin
           buffer.field1[0]:=read_g26(baseadr,axis,0);
           buffer.field1[1]:=read_g26(baseadr,axis,2);
           buffer.field1[2]:=read_g26(baseadr,axis,4);
           if (buffer.field1[2] and $8000)=$8000 then
              buffer.field1[3]:=$FFFF
           else
              buffer.field1[3]:=0;
         end;
     1 : begin
           buffer.field1[0]:=read_g26(baseadr,axis,6);
           buffer.field1[1]:=read_g26(baseadr,axis,8);
           buffer.field1[2]:=read_g26(baseadr,axis,10);
           if (buffer.field1[2] and $8000)=$8000 then
             buffer.field1[3]:=$FFFF
           else
             buffer.field1[3]:=0;
         end;
     end;

     if dir then
       begin
         buffer.field1[0]:=not(buffer.field1[0]);
         buffer.field1[1]:=not(buffer.field1[1]);
         buffer.field1[2]:=not(buffer.field1[2]);
         buffer.field1[3]:=not(buffer.field1[3]);
       end;

     read_count_value48:=buffer.field0;
  END;

{-------------------------------------------------------
  		    read_count_value32
 -------------------------------------------------------
 This function reads 32 bits of a measured value.
 -------------------------------------------------------}

FUNCTION
 read_count_value32(baseadr:word;axis,reg:byte;dir:boolean):comp;
  TYPE
    vartype = (li,by);
    mapper = record
               case wert:vartype of
                 li : (field0:longint);
                 by : (field1:array[0..1] of word);
               end;
  VAR
    buffer : mapper;
  BEGIN
    case reg of
     0 : begin
           buffer.field1[0]:=read_g26(baseadr,axis,0);
           buffer.field1[1]:=read_g26(baseadr,axis,2);
         end;
     1 : begin
           buffer.field1[0]:=read_g26(baseadr,axis,6);
           buffer.field1[1]:=read_g26(baseadr,axis,8);
         end;
     end;

     if dir then
       begin
         buffer.field1[0]:=not(buffer.field1[0]);
         buffer.field1[1]:=not(buffer.field1[1]);
       end;

     read_count_value32:=buffer.field0;
  END;

{-------------------------------------------------------
                         latched
 -------------------------------------------------------
 This function checks whether a measured value is
 latched in data register 0 or 1.
 -------------------------------------------------------}

FUNCTION latched(baseadr:word;axis,reg:byte):boolean;
  BEGIN
      case reg of
        0: latched:=
        (Read_g26(baseadr,axis,14) and $0001 ) = $0001;
        1: latched:=
        (Read_g26(baseadr,axis,14) and $0002 ) = $0002;
      end;
  END;

{-------------------------------------------------------
                       poll_latch
 -------------------------------------------------------
 This function polls until a measured value is
 latched in data register 0 or 1.
 -------------------------------------------------------}

PROCEDURE poll_latch(baseadr:word;axis,reg:byte);
  BEGIN
      case reg of
        0: begin
             repeat
             until latched(baseadr,axis,0);
           end;
        1: begin
             repeat
             until latched(baseadr,axis,1);
           end;
      end;
  END;
{-------------------------------------------------------}
END.