{-------------------------------------------------------}
{ DR. JOHANNES HEIDENHAIN GmbH, Traunreut, Germany      }
{                                                       }
{ Functions and Procedures to Adjust Pots of IK121      }
{                                                       }
{ V 1.01                                                }
{ April 1995                                            }
{ V 1.02                                                }
{ March 1996                                            }
{-------------------------------------------------------}
{ Reg. 0 stores poti value                              }
{ Reg. 2, 3 stores offset value of counter chip         }
{-------------------------------------------------------}

(*********************************************)
(* Low-Level Procedures: Bit Manipulations   *)
(*********************************************)
PROCEDURE wr_iic(scl,sda:boolean;board:ik121_pointr);
VAR
  buffer : word;
  BEGIN
    buffer:=g26_latch(board^.axis[2]);
    if not(scl) then buffer:=buffer or $8000;
    if not(sda) then buffer:=buffer or $0800;
    write_g26(board^.axis[2]^.baseadr,2,18,buffer);
  END;
(***************************************)
PROCEDURE scl0_sda0(board:ik121_pointr);
  BEGIN
    wr_iic(false,false,board);
  END;
(***************************************)
PROCEDURE scl1_sda0(board:ik121_pointr);
  BEGIN
    wr_iic(true,false,board);
  END;
(***************************************)
PROCEDURE scl0_sda1(board:ik121_pointr);
  BEGIN
    wr_iic(false,true,board);
  END;
(***************************************)
PROCEDURE scl1_sda1(board:ik121_pointr);
  BEGIN
    wr_iic(true,true,board);
  END;
(***************************************)
procedure delayer(x:word);
var i : word;
    a : byte;
begin
  for i:=1 to x do
    begin
      a:=port[$21];  (* I/O access is independent of PC clock *)
    end;
end;
(***************************************)
procedure start_iic(pointr:ik121_pointr);
begin
  scl0_sda1(pointr);
  delayer(10);
  scl1_sda1(pointr);
  delayer(10);
  scl1_sda0(pointr);
  delayer(10);
  scl0_sda0(pointr);
  delayer(10);
end;
(***************************************)
procedure stopp_iic(pointr:ik121_pointr);
begin
  scl0_sda0(pointr);
  delayer(10);
  scl1_sda0(pointr);
  delayer(10);
  scl1_sda1(pointr);
  delayer(10);
  scl0_sda1(pointr);
  delayer(5000);  (* V1.01 *)
end;
(***************************************)
function in_iic(pointr:ik121_pointr):boolean;
var
  buffer : word;
begin
  scl0_sda1(pointr);
  delayer(10);
  scl1_sda1(pointr);
  delayer(10);
  buffer:=read_g26(pointr^.axis[2]^.baseadr,2,14);      (* Read SO *)
  buffer:=buffer and $0020;
  scl1_sda1(pointr);
  delayer(10);
  scl0_sda1(pointr);
  delayer(10);
  if buffer=$0020 then in_iic:=true else in_iic:=false;
end;
(***************************************)
procedure out_0_iic(pointr:ik121_pointr);
begin
  scl0_sda0(pointr);
  delayer(10);
  scl1_sda0(pointr);
  delayer(10);
  scl0_sda0(pointr);
  delayer(10);
end;
(***************************************)
procedure out_1_iic(pointr:ik121_pointr);
begin
  scl0_sda1(pointr);
  delayer(10);
  scl1_sda1(pointr);
  delayer(10);
  scl0_sda1(pointr);
  delayer(10);
end;
(**************************************************)
(*   Medium-Level Procedures: Byte Manipulation   *)
(**************************************************)
procedure poll_poti(pointr:ik121_pointr;var error:boolean);
var
  bit           : boolean;
  timeout       : word;
begin
    timeout:=0;
    repeat
        start_iic(pointr);
        out_0_iic(pointr);
        out_1_iic(pointr);
        out_0_iic(pointr);
        out_1_iic(pointr);
        out_1_iic(pointr);   (* A3 *)
        out_1_iic(pointr);   (* A2 *)
        out_1_iic(pointr);   (* A1 *)
        out_1_iic(pointr);   (* A0 *)
        bit:=in_iic(pointr);
        if bit then stopp_iic(pointr);
        timeout:=timeout+1;
    until (not(bit)) or (timeout=100);
    error:=bit;
end;
(***************************************)
procedure poti_write2(pointr:ik121_pointr;instruct,poti,reg:byte;var error:boolean);
var
  bit           : byte;
begin
    poll_poti(pointr,error);
    if (not error) then
    begin
      bit:=instruct and $08;
      if bit=$08 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $04;
      if bit=$04 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=reg and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=reg and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      error:=error or in_iic(pointr);
    end;
  stopp_iic(pointr);
end;
(***************************************)
procedure poti_write3(pointr:ik121_pointr;instruct,poti,reg,data:byte;var error:boolean);
var
  bit            : byte;
begin
    poll_poti(pointr,error);
    if (not error) then
    begin
      bit:=instruct and $08;
      if bit=$08 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $04;
      if bit=$04 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=reg and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=reg and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      error:=error or in_iic(pointr);
      if (not error) then
      begin
        out_0_iic(pointr);
        out_0_iic(pointr);
        bit:=data and $20;
        if bit=$20 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $10;
        if bit=$10 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $08;
        if bit=$08 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $04;
        if bit=$04 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $02;
        if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $01;
        if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
        error:=error or in_iic(pointr);
      end;
    end;
 stopp_iic(pointr);
end;
(***************************************)
procedure poti_inc(pointr:ik121_pointr;poti,turns:byte;var error:boolean);
var
  bit,i : byte;
begin
    poll_poti(pointr,error);
    if (not error) then
    begin
      out_0_iic(pointr);    (* Increment/Decrement *)
      out_0_iic(pointr);
      out_1_iic(pointr);
      out_0_iic(pointr);
      bit:=poti and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      out_0_iic(pointr);
      out_0_iic(pointr);
      error:=error or in_iic(pointr);
      if (not error) then
      begin
        if turns>8 then turns:=8;
        for i:=1 to turns do
          begin
            out_1_iic(pointr);
          end;
      end;
    end;
 stopp_iic(pointr);
end;
(***************************************)
procedure poti_dec(pointr:ik121_pointr;poti,turns:byte;var error:boolean);
var
  bit,i : byte;
begin
    poll_poti(pointr,error);
    if (not error) then
    begin
      out_0_iic(pointr);    (* Increment/Decrement *)
      out_0_iic(pointr);
      out_1_iic(pointr);
      out_0_iic(pointr);
      bit:=poti and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      out_0_iic(pointr);
      out_0_iic(pointr);
      error:=error or in_iic(pointr);
      if (not error) then
      begin
        if turns>8 then turns:=8;
        for i:=1 to turns do
          begin
            out_0_iic(pointr);
          end;
      end;
    end;
 stopp_iic(pointr);
end;
(***************************************)
function poti_read(pointr:ik121_pointr;instruct,poti,reg:byte;var error:boolean):byte;
var
  bit,data       : byte;
  inbit          : boolean;
begin
    poll_poti(pointr,error);
    if (not error) then
    begin
      bit:=instruct and $08;
      if bit=$08 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $04;
      if bit=$04 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=instruct and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=poti and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=reg and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=reg and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      error:=error or in_iic(pointr);
      if (not error) then
      begin
        data:=0;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $80;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $40;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $20;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $10;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $08;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $04;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $02;
        inbit:=in_iic(pointr);
        if inbit then data:=data or $01;
        out_1_iic(pointr);      (* No acknowledge *)
      end;
    end;
 stopp_iic(pointr);
 poti_read:=data;
end;
(***************************************)
(*    High-Level Procedures            *)
(***************************************)
function Read_phasepoti(pointr:ik121_pointr;axis:byte;var error:boolean):byte;
var
  value:byte;
begin
  if (axis=1) then
    begin
      value:=poti_read(pointr,$09,$00,$00,error);
      (* Read register 0 pot 0 *)
    end
  else
   begin
     value:=poti_read(pointr,$09,$03,$00,error);
     (* Read register 0 pot 1*)
   end;
   Read_phasepoti:=value;
end;
(***************************************)
function Read_sympoti(pointr:ik121_pointr;axis:byte;var error:boolean):byte;
var
  value : byte;
begin
  if (axis=1) then
    begin
      value:=poti_read(pointr,$09,$01,$00,error);
      (* Read register 0 pot 2 *)
    end
  else
   begin
     value:=poti_read(pointr,$09,$02,$00,error);
     (* Read register 0 pot 3 *)
   end;
 Read_sympoti:=value;
end;
(***************************************)
procedure Write_phasepoti(pointr:ik121_pointr;axis,value:byte;var error:boolean);
var
  error0,error1 : boolean;
begin
  if (axis=1) then
    begin
      poti_write3(pointr,$0C,$00,$00,value,error0);
      poti_write2(pointr,$0D,$00,$00,error1);
      (* Write data register 0 pot 0 *)
    end
  else
    begin
      poti_write3(pointr,$0C,$03,$00,value,error0);
      poti_write2(pointr,$0D,$03,$00,error1);
      (* Write data register 0 pot 1 *)
    end;
  error:=error0 or error1;
end;
(***************************************)
procedure Write_sympoti(pointr:ik121_pointr;axis,value:byte;var error:boolean);
var
  error0,error1 : boolean;
begin
  if (axis=1) then
    begin
      poti_write3(pointr,$0C,$01,$00,value,error0);
      poti_write2(pointr,$0D,$01,$00,error1);
      (* Write data register 0 pot 2 *)
    end
  else
    begin
      poti_write3(pointr,$0C,$02,$00,value,error0);
      poti_write2(pointr,$0D,$02,$00,error1);
      (* Write data register 0 pot 3 *)
    end;
  error:=error0 or error1;
end;
(***************************************)
procedure Write_offset00(pointr:ik121_pointr;axis,value:integer);
begin
  if value<-63 then value:=-63;
  if value>63 then value:=63;
  pointr^.axis[axis]^.offset00:=value;
  write_offset(pointr^.axis[axis]^.baseadr,
               pointr^.axis[axis]^.axis,
               pointr^.axis[axis]^.offset00,
               pointr^.axis[axis]^.offset90);
end;
(***************************************)
procedure Write_offset90(pointr:ik121_pointr;axis,value:integer);
begin
  if value<-63 then value:=-63;
  if value>63 then value:=63;
  pointr^.axis[axis]^.offset90:=value;
  write_offset(pointr^.axis[axis]^.baseadr,
               pointr^.axis[axis]^.axis,
               pointr^.axis[axis]^.offset00,
               pointr^.axis[axis]^.offset90);
end;
(***************************************)
procedure Turn_phasepoti(pointr:ik121_pointr;axis,turns:byte;updown:boolean;var error:boolean);
begin
  if (axis=1) then
    begin
      if updown then poti_inc(pointr,0,turns,error) else poti_dec(pointr,0,turns,error);
    end
  else
   begin
      if updown then poti_inc(pointr,3,turns,error) else poti_dec(pointr,3,turns,error);
   end;
end;
(***************************************)
procedure Turn_sympoti(pointr:ik121_pointr;axis,turns:byte;updown:boolean;var error:boolean);
begin
  if (axis=1) then
    begin
      if updown then poti_inc(pointr,1,turns,error) else poti_dec(pointr,1,turns,error);
    end
  else
   begin
      if updown then poti_inc(pointr,2,turns,error) else poti_dec(pointr,2,turns,error);
   end;
end;
(***************************************)
procedure Turn_offsetdg00(pointr:ik121_pointr;axis,turns:byte;updown:boolean);
var i : byte;
begin
  for i:=1 to turns do
    begin
      if updown then
        begin
          if (pointr^.axis[axis]^.offset00<63) then
            begin
              pointr^.axis[axis]^.offset00:=pointr^.axis[axis]^.offset00+1
            end;
        end
      else
        begin
          if (pointr^.axis[axis]^.offset00>-63) then
            begin
              pointr^.axis[axis]^.offset00:=pointr^.axis[axis]^.offset00-1;
            end;
        end;
    end;
   write_offset(pointr^.axis[axis]^.baseadr,
                pointr^.axis[axis]^.axis,
                pointr^.axis[axis]^.offset00,
                pointr^.axis[axis]^.offset90);
end;
(***************************************)
procedure Turn_offsetdg90(pointr:ik121_pointr;axis,turns:byte;updown:boolean);
var i : byte;
begin
  for i:=1 to turns do
    begin
      if updown then
        begin
          if (pointr^.axis[axis]^.offset90<63) then
            begin
              pointr^.axis[axis]^.offset90:=pointr^.axis[axis]^.offset90+1
            end;
        end
      else
        begin
          if (pointr^.axis[axis]^.offset90>-63) then
            begin
              pointr^.axis[axis]^.offset90:=pointr^.axis[axis]^.offset90-1;
            end;
        end;
    end;
  write_offset(pointr^.axis[axis]^.baseadr,
               pointr^.axis[axis]^.axis,
               pointr^.axis[axis]^.offset00,
               pointr^.axis[axis]^.offset90);
end;
(***************************************)
procedure store_potis(pointr:ik121_pointr;var error:boolean);
begin
  (* Global write to all wiper registers in register 0 *)
  poti_write2(pointr,$08,$00,$00,error);
end;
(***************************************)



(*******************************************)
(*   Default Values for all Pot Registers  *)
(*******************************************)
procedure Poti_default(pointr:ik121_pointr;var error:boolean);
var
  err : boolean;
begin
  (* Pot in neutral position *)

  Write_phasepoti(pointr,1,32,error);
  Write_phasepoti(pointr,2,32,err);
  error:=error or err;
  Write_sympoti(pointr,1,32,err);
  error:=error or err;
  Write_sympoti(pointr,2,32,err);
  error:=error or err;

  (* Reset register 1,2,3 of pots 0,1,2,3 *)

  poti_write3(pointr,$0C,$00,$01,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$00,$02,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$00,$03,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$01,$01,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$01,$02,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$01,$03,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$02,$01,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$02,$02,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$02,$03,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$03,$01,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$03,$02,$00,err);
  error:=error or err;
  poti_write3(pointr,$0C,$03,$03,$00,err);
  error:=error or err;

end;
(***********************************************)
(*   Offset is only stored in EEPROM Registers *)
(***********************************************)
procedure Read_4_byte(pointr:ik121_pointr;var off0_1,off9_1,off0_2,off9_2:byte;var error:boolean);
var
  a   : byte;
  err : boolean;
begin
  (* Load offset from register 2 *)
  a:=poti_read(pointr,$0B,$00,$02,error);
  a:=a and $0F;
  off0_1:=a;
  a:=poti_read(pointr,$0B,$00,$03,err);
  error:=error or err;
  a:=a and $0F;
  off0_1:=off0_1+(a shl 4);

  a:=poti_read(pointr,$0B,$01,$02,err);
  error:=error or err;
  a:=a and $0F;
  off9_1:=a;
  a:=poti_read(pointr,$0B,$01,$03,err);
  error:=error or err;
  a:=a and $0F;
  off9_1:=off9_1+(a shl 4);

  a:=poti_read(pointr,$0B,$02,$02,err);
  error:=error or err;
  a:=a and $0F;
  off0_2:=a;
  a:=poti_read(pointr,$0B,$02,$03,err);
  error:=error or err;
  a:=a and $0F;
  off0_2:=off0_2+(a shl 4);

  a:=poti_read(pointr,$0B,$03,$02,err);
  error:=error or err;
  a:=a and $0F;
  off9_2:=a;
  a:=poti_read(pointr,$0B,$03,$03,err);
  error:=error or err;
  a:=a and $0F;
  off9_2:=off9_2+(a shl 4);

end;
(***************************************)
procedure Write_4_byte(pointr:ik121_pointr;off0_1,off9_1,off0_2,off9_2:byte;var error:boolean);
var
  a   : byte;
  err : boolean;
begin
  (* Offset wird in Register 2 gespeichert *)
  a:=off0_1 and $0F;
  poti_write3(pointr,$0C,$00,$02,a,error);
  a:=off0_1 shr 4;
  a:=a and $0F;
  poti_write3(pointr,$0C,$00,$03,a,err);
  error:=error or err;

  a:=off9_1 and $0F;
  poti_write3(pointr,$0C,$01,$02,a,err);
  error:=error or err;
  a:=off9_1 shr 4;
  a:=a and $0F;
  poti_write3(pointr,$0C,$01,$03,a,err);
  error:=error or err;

  a:=off0_2 and $0F;
  poti_write3(pointr,$0C,$02,$02,a,err);
  error:=error or err;
  a:=off0_2 shr 4;
  a:=a and $0F;
  poti_write3(pointr,$0C,$02,$03,a,err);
  error:=error or err;

  a:=off9_2 and $0F;
  poti_write3(pointr,$0C,$03,$02,a,err);
  error:=error or err;
  a:=off9_2 shr 4;
  a:=a and $0F;
  poti_write3(pointr,$0C,$03,$03,a,err);
  error:=error or err;

end;
(***************************************)
PROCEDURE Load_offset(board:IK121_pointr;var error:boolean);  (* Read EEPROM *)
  VAR
    a,b,c,d : byte;
    w       : integer;
  BEGIN
    Read_4_byte(board,a,b,c,d,error);
    if a<128 then
      begin
        w:=a;
      end
    else
      begin
        w:=not(a)+1;
        w:=-w;
      end;
    board^.axis[1]^.offset00:=w;
    if b<128 then
      begin
        w:=b;
      end
    else
      begin
        w:=not(b)+1;
        w:=-w;
      end;
    board^.axis[1]^.offset90:=w;
    if c<128 then
      begin
        w:=c;
      end
    else
      begin
        w:=not(c)+1;
        w:=-w;
      end;
    board^.axis[2]^.offset00:=w;
    if d<128 then
      begin
        w:=d;
      end
    else
      begin
        w:=not(d)+1;
        w:=-w;
      end;
    board^.axis[2]^.offset90:=w;
  END;
(***************************************)
PROCEDURE Store_offset(board:IK121_pointr;var error:boolean);  (* Write EEPROM *)
  VAR
    a,b,c,d : byte;
  BEGIN
    a:=lo(board^.axis[1]^.offset00);
    b:=lo(board^.axis[1]^.offset90);
    c:=lo(board^.axis[2]^.offset00);
    d:=lo(board^.axis[2]^.offset90);
    Write_4_byte(board,a,b,c,d,error);
  END;
(***************************************)
(*    IIC EEPROM Procedures            *)
(***************************************)
procedure poll_rom(pointr:ik121_pointr;var error:boolean);
var
  bit           : boolean;
  timeout       : word;
begin
    timeout:=0;
    repeat
        start_iic(pointr);
        out_1_iic(pointr);
        out_0_iic(pointr);
        out_1_iic(pointr);
        out_0_iic(pointr);
        out_0_iic(pointr);   (* A3 *)
        out_0_iic(pointr);   (* A2 *)
        out_0_iic(pointr);   (* A1 *)
        out_0_iic(pointr);   (* Write *)
        bit:=in_iic(pointr);
        stopp_iic(pointr);
        timeout:=timeout+1;
    until (not(bit)) or (timeout=100);
    error:=bit;
end;
(***************************************)
procedure rom_write(pointr:ik121_pointr;adr:word;data :byte;var error :boolean);
var
  bit             : word;
begin

    poll_rom(pointr,error);

    start_iic(pointr);
    out_1_iic(pointr);
    out_0_iic(pointr);
    out_1_iic(pointr);
    out_0_iic(pointr);
    out_0_iic(pointr);   (* A2 *)
    out_0_iic(pointr);   (* A1 *)
    bit:=adr and $0100;  (* A0 *)
    if bit=$0100 then out_1_iic(pointr) else out_0_iic(pointr);
    out_0_iic(pointr);   (* Write *)
    error:=error or in_iic(pointr);
    if (not error) then
    begin
      bit:=adr and $80;
      if bit=$80 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=adr and $40;
      if bit=$40 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=adr and $20;
      if bit=$20 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=adr and $10;
      if bit=$10 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=adr and $08;
      if bit=$08 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=adr and $04;
      if bit=$04 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=adr and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=adr and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      error:=error or in_iic(pointr);
      if (not error) then
      begin
        bit:=data and $80;
        if bit=$80 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $40;
        if bit=$40 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $20;
        if bit=$20 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $10;
        if bit=$10 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $08;
        if bit=$08 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $04;
        if bit=$04 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $02;
        if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
        bit:=data and $01;
        if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
        error:=error or in_iic(pointr);
      end;
   end;
 stopp_iic(pointr);
end;
(***************************************)
function rom_read(pointr:ik121_pointr;adr:word;var error:boolean):byte;
var
  data             : byte;
  bit              : word;
  inbit            : boolean;
begin

    poll_rom(pointr,error);

    start_iic(pointr);
    out_1_iic(pointr);
    out_0_iic(pointr);
    out_1_iic(pointr);
    out_0_iic(pointr);
    out_0_iic(pointr);   (* A2 *)
    out_0_iic(pointr);   (* A1 *)
    bit:=adr and $0100;  (* A0 *)
    if bit=$0100 then out_1_iic(pointr) else out_0_iic(pointr);
    out_0_iic(pointr);   (* Write *)
    error:=error or in_iic(pointr);
    if (not error) then
    begin
      bit:=adr and $80;
      if bit=$80 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=adr and $40;
      if bit=$40 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=adr and $20;
      if bit=$20 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=adr and $10;
      if bit=$10 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=adr and $08;
      if bit=$08 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=adr and $04;
      if bit=$04 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=adr and $02;
      if bit=$02 then out_1_iic(pointr) else out_0_iic(pointr);
      bit:=adr and $01;
      if bit=$01 then out_1_iic(pointr) else out_0_iic(pointr);
      error:=error or in_iic(pointr);
      if (not error) then
      begin
        start_iic(pointr);
        out_1_iic(pointr);
        out_0_iic(pointr);
        out_1_iic(pointr);
        out_0_iic(pointr);
        out_0_iic(pointr);   (* A2 *)
        out_0_iic(pointr);   (* A1 *)
        out_0_iic(pointr);   (* A0 *)
        out_1_iic(pointr);   (* Read *)
        error:=error or in_iic(pointr);
        if not(error) then
        begin
         data:=0;
         inbit:=in_iic(pointr);
         if inbit then data:=data or $80;
         inbit:=in_iic(pointr);
         if inbit then data:=data or $40;
         inbit:=in_iic(pointr);
         if inbit then data:=data or $20;
         inbit:=in_iic(pointr);
         if inbit then data:=data or $10;
         inbit:=in_iic(pointr);
         if inbit then data:=data or $08;
         inbit:=in_iic(pointr);
         if inbit then data:=data or $04;
         inbit:=in_iic(pointr);
         if inbit then data:=data or $02;
         inbit:=in_iic(pointr);
         if inbit then data:=data or $01;
         out_1_iic(pointr);      (* No acknowledge *)
       end;
     end;
   end;
 stopp_iic(pointr);
 rom_read:=data;
end;
(***************************************)
(*      IIC End                        *)
(***************************************)
