/*----------------------------IK121_1.CPP--------------------------------------

  DR. JOHANNES HEIDENHAIN GmbH, Traunreut, Germany

  Basic Functions for Ik121

  V 1.00
  April 1996
  V 1.01
  Mai 2000
  IC code for G38 and G49 added (ES)
 ---------------------------------------------------------------------------*/

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include "Ik121_1.h"


/*---------------------------------------------------------------------------
								InitIk121
 ----------------------------------------------------------------------------
 This function initializes two counters from one interface card IK121
 -------------------------Parameters-----------------------------------------
 Default parameters	
 ---------------------------------------------------------------------------*/
void InitIk121(pIk121 card)
{
	extern unsigned short Card_Adr;
	
	InitAxis(card->axis[1]);
	InitAxis(card->axis[2]);
	card->axis[1]->baseadr = Card_Adr;
	card->axis[2]->baseadr = Card_Adr;
	card->axis[1]->ucaxis = 1;
	card->axis[2]->ucaxis = 2;
	LoadOffset(card);
	InitHandler(card->axis[1]);
	InitHandler(card->axis[2]);
	CommHandler(card->axis[1]);
	CommHandler(card->axis[2]);
}

/*---------------------------------------------------------------------------
								InitAxis
 ----------------------------------------------------------------------------
 This function initializes one counter from one interface card IK121
 -------------------------Parameters-----------------------------------------
 Default parameters
 ---------------------------------------------------------------------------*/
void InitAxis(pG26 axis)
{
	axis->baseadr = 0x0330;
	axis->register0 = 0;
	axis->register1 = 0;

	axis->inimain.mode1024 = on;
	axis->inimain.en_timer = off;
	axis->inimain.en_48bit = on;
	axis->inimain.onefold = on;
	axis->inimain.twofold = off;
	axis->inimain.fourfold = off;
	axis->inimain.arcmode = off;
	axis->inimain.arc180000 = off;
	axis->inimain.sel_ri_1st = off;
	axis->inimain.sel_ri_2nd = off;

	axis->iniint.reg0 = off;
	axis->iniint.reg1 = off;
	axis->iniint.timer = off;
	axis->iniint.port = off;

	axis->inisync.en_l0_axis2 = off;
	axis->inisync.en_latch2_axis2 = off;
	axis->inisync.en_timer_axis2 = off;

	axis->inilatch.en_l0_reg0 = off;
	axis->inilatch.en_latch2_reg0 = off;
	axis->inilatch.en_timer_reg0 = off;
	axis->inilatch.en_l1_reg1 = on;
	axis->inilatch.en_latch2_reg1 = off;
	axis->inilatch.en_timer_reg1 = off;

	axis->softcomm.start = on;
	axis->softcomm.stop = off;
	axis->softcomm.clear = on;
	axis->softcomm.latch0 = off;
	axis->softcomm.latch1 = off;
	axis->softcomm.latch2 = off;
	axis->softcomm.clrfreq = on;
	axis->softcomm.clrstat = on;

	axis->refcomm.ristart = off;
	axis->refcomm.ristop = off;
	axis->refcomm.riclear = off;
	axis->refcomm.riclear2 = off;
	axis->refcomm.rilatch = off;
	axis->refcomm.rilatch2 = off;

	axis->cstatus.latch0 = off;
	axis->cstatus.latch1 = off;
	axis->cstatus.stop = off;
	axis->cstatus.sda = off;
	axis->cstatus.error_freq = off;
	axis->cstatus.ref_activ = off;

	axis->sstatus.ad00 = 0;
	axis->sstatus.ad90 = 0;
	axis->sstatus.amp_act = 3;
	axis->sstatus.amp_min = 3;

	axis->timer = 0xFFFF;
	axis->offset00 = 0;
	axis->offset90 = 0;
}

/*---------------------------------------------------------------------------
								InitHandler
 ----------------------------------------------------------------------------
 This function writes the datas for the initialization from the 
 RAM memory model to the registers
 --------------------------------------------------------------------------*/
void InitHandler(pG26 axis)
{                                              
unsigned short lat_buffer,syn_buffer,betr_buffer,int_buffer,timer_buffer;
				  
	// Fixed Values
	WriteG26(axis->baseadr,axis->ucaxis,CONTROLREG3,0x0000);
	WriteG26(axis->baseadr,axis->ucaxis,CONTROLREG2,0x00c8);

	// Main 
	betr_buffer = G26Main(axis);
	WriteG26(axis->baseadr,axis->ucaxis,INITIALREG,betr_buffer);

	// Latch and sync 
  lat_buffer = G26Latch(axis);
  syn_buffer = G26Sync(axis);
  WriteG26(axis->baseadr,axis->ucaxis,ENDATUMREG,lat_buffer | syn_buffer);

	// Interrupt
  int_buffer = G26Int(axis);
  WriteG26(axis->baseadr,axis->ucaxis,ENINTREG,int_buffer);

	// Timer Value
  timer_buffer = axis->timer;
  WriteG26(axis->baseadr,axis->ucaxis,TIMERREG,timer_buffer);

	// Offset DG00,DG90
  WriteOff00(axis->baseadr,axis->ucaxis,axis->offset00);
  WriteOff90(axis->baseadr,axis->ucaxis,axis->offset90);
}

/*---------------------------------------------------------------------------
								CommHandler
 ----------------------------------------------------------------------------
 This function writes the datas from the RAM memory model to the registers
 --------------------------------------------------------------------------*/
void CommHandler(pG26 axis)
{
	unsigned short	 soft_buffer,ref_buffer;

	soft_buffer = G26Soft(axis);
	if (soft_buffer != 0)
		 WriteG26(axis->baseadr,axis->ucaxis,CONTROLREG1,soft_buffer);
	ref_buffer = G26Ref(axis);
	 if (ref_buffer != 0)
		 WriteG26(axis->baseadr,axis->ucaxis,REFMARKREG,ref_buffer);
}

/*---------------------------------------------------------------------------
								G26Main
 ----------------------------------------------------------------------------
 This function defines default values
 --------------------------------------------------------------------------*/
unsigned short G26Main(pG26 axis)
{
	unsigned short	 betr_buffer = 0x0000;
	if (axis->inimain.mode1024) {betr_buffer |= 0x0001;}
	if (axis->inimain.en_timer) {betr_buffer |= 0x0004;}
	if (axis->inimain.en_48bit) {betr_buffer |= 0x0040;}
	if (axis->inimain.fourfold)
		{
		axis->inimain.onefold = off;
		axis->inimain.twofold = off;
		betr_buffer |= 0x0200;
		betr_buffer |= 0x0300;
		}
	if (axis->inimain.twofold)
		{
		axis->inimain.onefold = off;
		axis->inimain.fourfold = off;
		betr_buffer |= 0x0300;
		}
	 if (axis->inimain.onefold)
		{
		axis->inimain.twofold = off;
		axis->inimain.fourfold = off;
		}
	 if (axis->inimain.arcmode) {betr_buffer |= 0x0400;}
	 if (axis->inimain.arc180000) {betr_buffer |= 0x0800;}
	 if (axis->inimain.sel_ri_1st) {betr_buffer |= 0x4000;}
	 if (axis->inimain.sel_ri_2nd) {betr_buffer |= 0x8000;}

	 return(betr_buffer);
}

/*---------------------------------------------------------------------------
								G26Latch
 ----------------------------------------------------------------------------
 This function sets the latch registers for the specified axis 
 --------------------------------------------------------------------------*/
unsigned short G26Latch(pG26 axis)
{
	unsigned short latch_buffer = 0;
  
	if (axis->inilatch.en_l0_reg0) {latch_buffer |= 0x0001;}
	if (axis->inilatch.en_latch2_reg0) {latch_buffer |= 0x0004;}
	if (axis->inilatch.en_timer_reg0)  {latch_buffer |= 0x0008;}
		// L1 slave only 
	if (axis->inilatch.en_l1_reg1) {latch_buffer |= 0x0010;}
	if (axis->inilatch.en_latch2_reg1) {latch_buffer |= 0x0040;}
	if (axis->inilatch.en_timer_reg1) {latch_buffer |= 0x0080;}
	return(latch_buffer);
}
 
/*---------------------------------------------------------------------------
								G26Sync
 ----------------------------------------------------------------------------
 This function sets the axis cascading and the IIC registers 
 for the specified axis 
 --------------------------------------------------------------------------*/
unsigned short G26Sync(pG26 axis)
{
	unsigned short syn_buffer = 0x0000;

	if (axis->ucaxis == 1)
	{
	if (axis->inisync.en_l0_axis2) {syn_buffer |= 0x0100;}
	if (axis->inisync.en_latch2_axis2) {syn_buffer |= 0x0200;}
	if (axis->inisync.en_timer_axis2) {syn_buffer |= 0x0400;}
	}
	else;								// sda=1, scl=1
	return(syn_buffer);
}

/*---------------------------------------------------------------------------
								G26Int
 ----------------------------------------------------------------------------
 This function sets the interrupt registers for the specified axis 
 --------------------------------------------------------------------------*/
unsigned short G26Int(pG26 axis)
{
	unsigned short int_buffer = 0x0000;

	if (axis->iniint.reg0) {int_buffer |= 0x0001;}
	if (axis->iniint.reg1) {int_buffer |= 0x0002;}
	if (axis->iniint.timer) {int_buffer |= 0x0004;}
	if (axis->iniint.port) {int_buffer |= 0x0008;}
	return(int_buffer);
}
 
/*---------------------------------------------------------------------------
								G26Soft
 ----------------------------------------------------------------------------
 This function sets the software latch registers for the specified axis 
 --------------------------------------------------------------------------*/
unsigned short G26Soft(pG26 axis)
{
	unsigned short control_buffer = 0x0000;
	if (axis->softcomm.start)
		{
		control_buffer |= 0x0008;
		axis->softcomm.start = off;
		}
	if (axis->softcomm.stop)
		{
		control_buffer |= 0x0010;
		axis->softcomm.stop = off;
		}
	if (axis->softcomm.clear)
		{
		control_buffer |= 0x0020;
		axis->softcomm.clear = off;
		}
	if (axis->softcomm.latch0)
		{
		control_buffer |= 0x0001;
		axis->softcomm.latch0 = off;
		}
	if (axis->softcomm.latch1)
		{
		control_buffer |= 0x0002;
		axis->softcomm.latch1 = off;
		}
	if (axis->softcomm.latch2)
		{
		control_buffer |= 0x0004;
		axis->softcomm.latch2 = off;
		}
	if (axis->softcomm.clrfreq)
		{
		control_buffer |= 0x0040;
		axis->softcomm.clrfreq = off;
		}
	if (axis->softcomm.clrstat)
		{
		control_buffer |= 0x0080;
		axis->softcomm.clrstat = off;
		}
	return(control_buffer);
}

/*---------------------------------------------------------------------------
								G26Ref
 ----------------------------------------------------------------------------
 This function sets the reference mark register for the specified axis 
 --------------------------------------------------------------------------*/
unsigned short G26Ref(pG26 axis)
{
	unsigned short	 ref_buffer = 0x0000;

	if (axis->refcomm.ristart)
		{
		ref_buffer |= 0x0001;
		axis->refcomm.ristart = off;
		}
	if (axis->refcomm.ristop)
		{
		ref_buffer |= 0x0002;
		axis->refcomm.ristop = off;
		}
	if (axis->refcomm.riclear)
		{
		ref_buffer |= 0x0004;
		axis->refcomm.riclear = off;
		}
	if (axis->refcomm.riclear2)
		{
		ref_buffer |= 0x0020;
		axis->refcomm.riclear2 = off;
		}
	if (axis->refcomm.rilatch)
		{
		ref_buffer = ref_buffer |= 0x0008;
		axis->refcomm.rilatch = off;
		}
	if (axis->refcomm.rilatch2)
		{
		ref_buffer |= 0x0010;
		axis->refcomm.rilatch2 = off;
		}
	return(ref_buffer);
}

/*---------------------------------------------------------------------------
								ReadCountStatus
 ----------------------------------------------------------------------------
 This function reads the counter status for the specified axis 
 --------------------------------------------------------------------------*/
void ReadCountStatus(pG26 axis)
{
	unsigned short buffer;

	buffer = ReadG26(axis->baseadr,axis->ucaxis,STATUSREG12);
	if (buffer & 0x0001)
			axis->cstatus.latch0 = on;
	else	axis->cstatus.latch0 = off;
	if (buffer & 0x0002)
			axis->cstatus.latch1 = on;
	else	axis->cstatus.latch1 = off;
	if (buffer & 0x0010)
			axis->cstatus.stop = on;
	else	axis->cstatus.stop = off;
	if (buffer & 0x0020)
			axis->cstatus.sda = on;
	else	axis->cstatus.sda = off;
	if (buffer & 0x0040)
			axis->cstatus.error_freq = on;
	else	axis->cstatus.error_freq = off;
	if (buffer & 0x0100)
			axis->cstatus.ref_activ = on;
	else	axis->cstatus.ref_activ = off;
}

/*---------------------------------------------------------------------------
								ReadSignalStatus
 ----------------------------------------------------------------------------
 This function reads the signal status for the specified axis 
 --------------------------------------------------------------------------*/
void ReadSignalStatus(pG26 axis)
{
	unsigned short buffer = 0, timeout = 0;
	
	WriteG26(axis->baseadr,axis->ucaxis,CONTROLREG2,0x00d8);
	do
		{
		timeout += 1;
		buffer = ReadG26(axis->baseadr,axis->ucaxis,STATUSREG3);
		}
	while (!(buffer & 0x0010) && (timeout <= 4095));
	axis->sstatus.ad00 = ReadG26(axis->baseadr,axis->ucaxis,AMPLITUDE00);
	axis->sstatus.ad90 = ReadG26(axis->baseadr,axis->ucaxis,AMPLITUDE90);
	buffer = ReadG26(axis->baseadr,axis->ucaxis,AMPLITUDEREG);
	axis->sstatus.amp_act = (buffer & 0x0300) >>8;
	axis->sstatus.amp_min = (buffer & 0x0c00) >>10;
	WriteG26(axis->baseadr,axis->ucaxis,CONTROLREG2,0x00c8);
	WriteG26(axis->baseadr,axis->ucaxis,CONTROLREG1,0x0080);
}

/*---------------------------------------------------------------------------
								ReadIntStatus
 ----------------------------------------------------------------------------
 This function reads the interrupt status for the specified axis 
 --------------------------------------------------------------------------*/
void ReadIntStatus(pG26 axis)
{
	unsigned short buffer;
	
	buffer = ReadG26(axis->baseadr,axis->ucaxis,INTSTATUSREG);

	if (buffer & 0x0100)
			axis->istatus.l0 = on;
	else
			axis->istatus.l0 = off;
	if (buffer & 0x0200)
			axis->istatus.l1 = on;
	else
			axis->istatus.l1 = off;
	if (buffer & 0x0400)
			axis->istatus.timer = on;
	else
			axis->istatus.timer = off;
	if (buffer & 0x0001)
			axis->istatus.pendl0 = on;
	else
			axis->istatus.pendl0 = off;
	if (buffer & 0x0002)
			axis->istatus.pendl1 = on;
	else
			axis->istatus.pendl1 = off;
	if (buffer & 0x0004)
			axis->istatus.pendtimer = on;
	else
			axis->istatus.pendtimer = off;
}
 
/*---------------------------------------------------------------------------
								LookforIk121
 ----------------------------------------------------------------------------
 This function is looking for an interface card IK121
 --------------------------------------------------------------------------*/
unsigned char LookforIk121(pIk121 card)
{
	unsigned short buffer1,buffer2;
	
	buffer1 = ReadG26(card->axis[1]->baseadr,1,STATUSREG3);
	buffer1 = (buffer1 & 0xff00);
	buffer1 = ( (buffer1==0x0800) || (buffer1==0x0900) || (buffer1==0x0D00) )? on : off;

	buffer2 = ReadG26(card->axis[1]->baseadr,2,STATUSREG3);
	buffer2 = (buffer2 & 0xff00);
	buffer2 = ( (buffer2==0x0800) || (buffer2==0x0900) || (buffer2==0x0D00) )? on : off;

	return(buffer1 & buffer2);
}

/*---------------------------------------------------------------------------
								ReadReg0
 ----------------------------------------------------------------------------
 This function reads the counter value from register 0 for the 
 specified axis into the RAM memory model 
 --------------------------------------------------------------------------*/
void ReadReg0(pG26 axis)
{
	double value;
	PollLatch(axis->baseadr,axis->ucaxis,0);
	if (axis->inimain.en_48bit)
		value = ReadCountValue_48(axis->baseadr,axis->ucaxis,0);
	else
		value = (double)ReadCountValue_32(axis->baseadr,axis->ucaxis,0);

	if (axis->inimain.mode1024)
		axis->register0 = value;
	else
		axis->register0 = value/1024;
}

/*---------------------------------------------------------------------------
								ReadReg1
 ----------------------------------------------------------------------------
 This function reads the counter value from register 1 for the 
 specified axis into the RAM memory model 
 --------------------------------------------------------------------------*/
void ReadReg1(pG26 axis)
{
	double value;

	PollLatch(axis->baseadr,axis->ucaxis,1);                       
	if (axis->inimain.en_48bit)
		value = ReadCountValue_48(axis->baseadr,axis->ucaxis,1);
	else
		value = (double)ReadCountValue_32(axis->baseadr,axis->ucaxis,1);

	if (axis->inimain.mode1024)
		axis->register1 = value;
	else
		axis->register1 = value/1024;
}

/*---------------------------------------------------------------------------
								WriteOff00
 ----------------------------------------------------------------------------
 This function writes the offset of the 0 degree encoder signal from the 
 RAM memory model into the EEPROM
 --------------------------------------------------------------------------*/
void WriteOff00(unsigned short baseadr,unsigned char axis,short off00)
{
	unsigned short timeout,buffer,off0;
	timeout = 0;
	
	do
		{
		timeout += 1;
		buffer = ReadG26(baseadr,axis,STATUSREG3);
		}
	while((buffer & 0x0020) && (timeout < 1000));
	if (off00 < 0)
		{
		if (off00 < -63) off00 = -63;
		off00 += 127;
		off0 = off00 | 0x0040;	// Minus
		}
	else
		{
		if (off00 > 63) off00 = 63;
		off0 = off00;
		}
	 WriteG26(baseadr,axis,OFFSET00,off0);
}
 
/*---------------------------------------------------------------------------
								WriteOff90
 ----------------------------------------------------------------------------
 This function writes the offset of the 90 degree encoder signal from the 
 RAM memory model into the EEPROM
 --------------------------------------------------------------------------*/
void WriteOff90(unsigned short baseadr,unsigned char axis,short off90)
{
	unsigned short timeout,buffer,off9;
	timeout = 0;
	
	do
		{
		timeout += 1;
		buffer = ReadG26(baseadr,axis,STATUSREG3);
		}
	while((buffer & 0x0040) && (timeout < 1000));
	if (off90 < 0)
		{
		if (off90 < -63) off90 = -63;
		off90 += 127;
		off9 = off90 | 0x0040;	// Minus
		}
	else
		{
		if (off90 > 63) off90 = 63;
		off9 = off90;
		}
	 WriteG26(baseadr,axis,OFFSET90,off9);
}
 
/*--------------------------------------------------------------------------
								WriteG26
 ---------------------------------------------------------------------------
 This function writes a value in a 16-bit
 register of a counter.
 -------------------------Parameters-----------------------------------------
 baseadr   basic address A0 to A9 of the IK 121
 axis	     axis select - axis 1 or axis 2
 address   address of the counter B0 to B4
 datum	  value to write to the address
 adr_reg   address of the address register on the IK 121
 adr_point the part of the address indicated by the
			  address register (R0 bis R2)
 adr_gate  port address in which <datum> is written
 --------------------------------------------------------------------------*/
void WriteG26 (unsigned short baseadr, unsigned char axis,
					unsigned short address, unsigned short datum)
{
unsigned short         adr_reg, adr_point, adr_gate;
	/* Discard the last four bits of the card
	address */
baseadr = baseadr & 0x0FF0;
	/* Address B0 to B4 of the counter */
address = address & 0x1F;

/* Load address pointer into address register */
	/* Adress of the address register */
adr_reg = baseadr | 0x0008;
	/* Value of the address register R0 to R2 =
	address of the counter without B0 & B1 */
adr_point = address >> 2;
	/* Load address register */
outpw (adr_reg, adr_point);

/* Calculate port address */
switch (axis)
		 {
		 case 1:
		 adr_gate = baseadr | (address & 0x03);
		 break;
		 case 2:
		 adr_gate = (baseadr | 0x0004) | (address & 0x03);
		 break;
		 default:
		 break;
		 }

	/* Write <datum> to the counter */
outpw (adr_gate, datum);
}

/*---------------------------------------------------------------------------
							  ReadG26
 ----------------------------------------------------------------------------
 This function reads a value from a 16-bit
 register of a counter.
 -----------------------Parameters-------------------------------------------
 baseadr   basic address A0 to A9 of the IK 121
 axis      axis select - axis 1 or axis 2
 address   address of the counter B0 to B4
 adr_reg   address of the address register on the IK 121
 adr_point the part of the address indicated by the
			  address register (R0 bis R2)
 adr_gate  port address in which <datum> is written
 --------------------------------------------------------------------------*/
unsigned int ReadG26(unsigned short baseadr, unsigned char axis,
							  unsigned short address)
{
unsigned short adr_reg, adr_point, adr_gate;
	/* Discard the last four bits of the card address */
baseadr = baseadr & 0x0FF0;
	/* Address B0 to B4 of the counter */
address = address & 0x1F;

/* Load address pointer into address register */
	/* Address of the address register */
adr_reg = baseadr | 0x0008;
	/* Value of the address register R0 to R2 =
	address of the counter without B0 and B1 */
adr_point = address >> 2;
	/* Load address register */
outpw (adr_reg, adr_point);

/* Calculate port address */
switch (axis)
		 {
		 case 1:
		 adr_gate = baseadr | (address & 0x03);
		 break;
		 case 2:
		 adr_gate = (baseadr | 0x0004) | (address & 0x03);
		 break;
		 default:
		 break;
		 }                                            
	/* Read datum from the counter */
return(inpw(adr_gate));
}

/*---------------------------------------------------------------------------
								 Latched
 ----------------------------------------------------------------------------
 This function checks whether a measured value is
 latched in data register 0 or 1.
 ---------------------------------------------------------------------------*/
unsigned char Latched(unsigned int baseadr,unsigned char axis,unsigned char reg)
{
unsigned char result;
switch (reg)
		 {
		 case 0:
		 result = (unsigned char)
					 (ReadG26 (baseadr, axis, STATUSREG12) & 0x01);
		 break;
		 case 1:
		 result = (unsigned char)
					 (ReadG26 (baseadr, axis, STATUSREG12) & 0x02);
		 break;
		 }
return (result);
}

/*---------------------------------------------------------------------------
							  PollLatch
 ----------------------------------------------------------------------------
 This function polls until a measured value is
 latched in data register 0 or 1.
 ---------------------------------------------------------------------------*/
void PollLatch(unsigned int baseadr,unsigned char axis,unsigned char reg)
{
switch (reg)
		 {
		 case 0:
		 while (Latched (baseadr, axis, 0) == 0)
		 ;
		 break;

		 case 1:
		 while (Latched (baseadr, axis, 1) == 0)
		 ;
		 break;
		 }
}
									  
/*---------------------------------------------------------------------------
						  ReadCountValue_48
 ----------------------------------------------------------------------------
 This function reads 48 bits of a measured value.
 ---------------------------------------------------------------------------*/
double ReadCountValue_48(unsigned short baseadr,unsigned char axis,
								 unsigned char reg)
{
unsigned int field[3];
double count_value48;

switch (reg)
		 {
		 case 0:
		 field[0] = ReadG26 (baseadr, axis, DATUMREG0LSB);
		 field[1] = ReadG26 (baseadr, axis, DATUMREG0);
		 field[2] = ReadG26 (baseadr, axis, DATUMREG0MSB);
		 break;
		 case 1:
		 field[0] = ReadG26 (baseadr, axis, DATUMREG1LSB);
		 field[1] = ReadG26 (baseadr, axis, DATUMREG1);
		 field[2] = ReadG26 (baseadr, axis, DATUMREG1MSB);
		 break;
		 }

if (field[2] & 0x8000)
		 count_value48 = (double)((field[0]-65535.0) +
		 65536.0*(field[1]-65535.0)+
		 4294967296.0*(field[2]-65535.0)-1);
		 else
		 count_value48 = (double)(field[0] +
		 65536.0*field[1] +
		 4294967296.0*field[2]);


return (count_value48);
}
/*--------------------------------------------------------------------------
						  ReadCountValue_32
 ---------------------------------------------------------------------------
 This function reads 32 bits of a measured value.
 ---------------------------------------------------------------------------*/
long ReadCountValue_32(unsigned short baseadr,unsigned char axis,
							  unsigned  char reg)
{
union 	mapper
		  {
		  long field0;
		  unsigned int field1[2];
		  }buffer;

switch (reg)
		 {
		 case 0:
		 buffer.field1[0] = ReadG26 (baseadr, axis, DATUMREG0LSB);
		 buffer.field1[1] = ReadG26 (baseadr, axis, DATUMREG0);
		 break;
		 case 1:
		 buffer.field1[0] = ReadG26 (baseadr, axis, DATUMREG1LSB);
		 buffer.field1[1] = ReadG26 (baseadr, axis, DATUMREG1);
		 break;
		 }
return (buffer.field0);
}


