/*-----------------------------IIC.CPP--------------------------------------*/
/*                                                                          */
/*			DR. JOHANNES HEIDENHAIN GmbH, Traunreut, Germany                   */
/*                                                                          */
/*			Functions to Adjust Pots of Ik121                                  */
/*                                                                          */
/*			V 1.00                                                             */
/*			April 1996                                                         */
/*--------------------------------------------------------------------------*/
/*			The values of the amplitude and phase shift of the encoder signals */
/*       are stored in register 0 of the EEPROM.                            */
/*			The offset values of the encoder signals are stored in             */
/*       register 2 and 3 of the EEPROM.                                    */
/*--------------------------------------------------------------------------*/


#include <conio.h>
#include "Ik121_1.h"                   


/****************************************************************************/
/**		Globals                                                           **/
/****************************************************************************/
unsigned char iic_error;


/****************************************************************************/
/**		Low-Level Functions: Bit Manipulations                            **/
/****************************************************************************/
void WrIic(unsigned char scl, unsigned char sda, pIk121 card)
{
	unsigned short  buffer;

	buffer = G26Latch(card->axis[2]);          
	if(!scl) buffer |= 0x8000;
	if(!sda) buffer |= 0x0800;
	WriteG26(card->axis[2]->baseadr,2,ENDATUMREG,buffer);
}

/****************************************************************************/
void Scl0_Sda0(pIk121 card)
{
	 WrIic(off,off,card);
}

/****************************************************************************/
void Scl1_Sda0(pIk121 card)
{
	 WrIic(on,off,card);
}

/****************************************************************************/
void Scl0_Sda1(pIk121 card)
{
	 WrIic(off,on,card);
}

/****************************************************************************/
void Scl1_Sda1(pIk121 card)
{
	 WrIic(on,on,card);
}

/****************************************************************************/
void delay_us(unsigned short us)
	{
	int l,h;
	long summ,c,d,e,max;
	max=us*2380L/1000l;
	summ=0L;
	outp(67,0);
	l=inp(64);
	h=inp(64); 
	d=l+(h<<8);
	while(summ<max)
		{
		outp(67,0);
		l=inp(64);
		h=inp(64); 
		c=l+(h<<8);
		e=d-c;
		if(e<0L)e=e+65535L;		// Overflow
		summ=summ+e;
		d=c;
		}
	 }

/****************************************************************************/
void StartIic(pIk121 card)
{
	Scl1_Sda1(card);
	delay_us(10);
	Scl1_Sda0(card);
	delay_us(10);
	Scl0_Sda0(card);
	delay_us(10);
}

/****************************************************************************/
void StopIic(pIk121 card)
{
	Scl0_Sda0(card);
	delay_us(10);
	Scl1_Sda0(card);
	delay_us(10);
	Scl1_Sda1(card);
	delay_us(10000);
}

/****************************************************************************/
unsigned char InIic(pIk121 card)
{
	unsigned short buffer;
	Scl0_Sda1(card);
	delay_us(10);
	Scl1_Sda1(card);
	delay_us(10);
	buffer = ReadG26(card->axis[2]->baseadr,2,STATUSREG12);      // Read SO 
	buffer &= 0x0020;
	Scl1_Sda1(card);
	delay_us(10);
	Scl0_Sda1(card);
	delay_us(10);
	if (buffer == 0x0020) 
		return(on);
	else
		return(off);
}

/****************************************************************************/
void Out_0_Iic(pIk121 card)
{
  Scl0_Sda0(card);
  delay_us(10);
  Scl1_Sda0(card);
  delay_us(10);
  Scl0_Sda0(card);
  delay_us(10);
}

/****************************************************************************/
void Out_1_Iic(pIk121 card)
{
  Scl0_Sda1(card);
  delay_us(10);
  Scl1_Sda1(card);
  delay_us(10);
  Scl0_Sda1(card);
  delay_us(10);
}

/****************************************************************************/
/**		Medium-Level Functions: Byte Manipulation                         **/
/****************************************************************************/
void PollPoti(pIk121 card)
{
	unsigned char  bit;
	unsigned short timeout = 0;
	
	do
		{
		StartIic(card);
		Out_0_Iic(card);
		Out_1_Iic(card);
		Out_0_Iic(card);
		Out_1_Iic(card);
		Out_1_Iic(card);		// A3
		Out_1_Iic(card);   	// A2
		Out_1_Iic(card);   	// A1 
		Out_1_Iic(card);   	// A0 
		bit = InIic(card);
		if (bit) StopIic(card);
		timeout += 1;
		}
	while ((bit) && (timeout < 100));
	iic_error = bit;
}

/*****************************************************************************/
void PotiWrite2(pIk121 card, unsigned char instruct, unsigned char poti,
					unsigned char reg)
{
	PollPoti(card);
	if (!iic_error)
		{
		if (instruct & 0x08) Out_1_Iic(card);	else Out_0_Iic(card);
		if (instruct & 0x04) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		if (reg & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (reg & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		iic_error |= InIic(card);
		}
  StopIic(card);
}

/*****************************************************************************/
void PotiWrite3(pIk121 card, unsigned char instruct, unsigned char poti,
					unsigned char reg, unsigned char data)
{
	PollPoti(card);
	
	if (!iic_error)
		{
		if (instruct & 0x08) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x04) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		if (reg & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (reg & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		iic_error |= InIic(card);
		if (!iic_error)
			{
			Out_0_Iic(card);
			Out_0_Iic(card);
			if (data & 0x20) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x10) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x08) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x04) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
			iic_error |= InIic(card);
			}
		}
	StopIic(card);
}

/*****************************************************************************/
void PotiInc(pIk121 card, unsigned char poti, unsigned char turns)
{
	PollPoti(card);
	if (!iic_error)
		{
		Out_0_Iic(card);    		// Decrement 
		Out_0_Iic(card);
		Out_1_Iic(card);
		Out_0_Iic(card);
		if (poti & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		Out_0_Iic(card);
		Out_0_Iic(card);
		iic_error |= InIic(card);
		if (!iic_error)
			{
			if (turns > 8) turns = 8;
			for (short i=1; i<=turns; i++)
				Out_1_Iic(card);
			}
		}
	StopIic(card);
}

/*****************************************************************************/
void PotiDec(pIk121 card, unsigned char poti, unsigned char turns)
{
	PollPoti(card);
	if (!iic_error)
		{
		Out_0_Iic(card);    		// Increment/Decrement
		Out_0_Iic(card);
		Out_1_Iic(card);
		Out_0_Iic(card);
		if (poti & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		Out_0_Iic(card);
		Out_0_Iic(card);
		iic_error |= InIic(card);
		if (!iic_error)
			{
			if (turns > 8) turns = 8;
			for (short i=1; i<=turns; i++)
				Out_0_Iic(card);
			}
		}
	StopIic(card);
}

/****************************************************************************/
unsigned char PotiRead(pIk121 card, unsigned char instruct, unsigned char poti,
								unsigned char reg)
{
	unsigned char inbit, data = 0;
	
	PollPoti(card);
	if (!iic_error)
		{
		if (instruct & 0x08) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x04) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		if (reg & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (reg & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		iic_error |= InIic(card);
		if (!iic_error)
			{
			inbit = InIic(card);
			if (inbit) data |= 0x80;
			inbit = InIic(card);
			if (inbit) data |= 0x40;
			inbit = InIic(card);
			if (inbit) data |= 0x20;
			inbit = InIic(card);
			if (inbit) data |= 0x10;
			inbit = InIic(card);
			if (inbit) data |= 0x08;
			inbit = InIic(card);
			if (inbit) data |= 0x04;
			inbit = InIic(card);
			if (inbit) data |= 0x02;
			inbit = InIic(card);
			if (inbit) data |= 0x01;
			Out_1_Iic(card);			      // No acknowledge
			}	
		}
 StopIic(card);
 return(data);
}

/****************************************************************************/
/**		High-Level Functions                                              **/
/****************************************************************************/
unsigned char ReadPhasepoti(pIk121 card, unsigned char axis)
{
	unsigned char value;

	switch(axis)
		{
		case(1):
		value = PotiRead(card,0x09,0x00,0x00);		// Read register 0 pot 0
		break; 
		case(2):
		value = PotiRead(card,0x09,0x03,0x00);		// Read register 0 pot 1
		break;
		default:
		break;
		}
	return(value);
}
/****************************************************************************/
unsigned char ReadSympoti(pIk121 card, unsigned char axis)
{
	unsigned char value;
	
		switch(axis)
		{
		case(1):
		value = PotiRead(card,0x09,0x01,0x00);		// Read register 0 pot 2
		break; 
		case(2):
		value = PotiRead(card,0x09,0x02,0x00);		// Read register 0 pot 3
		break;
		default:
		break;
		}
	return(value);
}

/****************************************************************************/
void WritePhasepoti(pIk121 card, unsigned char axis, unsigned char value)
{
	switch(axis)
		{
		case(1): 			// Write data register 0 pot 0
		PotiWrite3(card,0x0C,0x00,0x00,value);
		PotiWrite2(card,0x0D,0x00,0x00);
		break; 
		case(2):			  	// Write data register 0 pot 1
		PotiWrite3(card,0x0C,0x03,0x00,value);
		PotiWrite2(card,0x0D,0x03,0x00);
		break;
		default:
		break;
		}
}

/****************************************************************************/
void WriteSympoti(pIk121 card, unsigned char axis, unsigned char value)
{
	switch(axis)
		{
		case(1): 			// Write data register 0 pot 2
		PotiWrite3(card,0x0C,0x01,0x00,value);
		PotiWrite2(card,0x0D,0x01,0x00);
		break; 
		case(2):			  	// Write data register 0 pot 3
		PotiWrite3(card,0x0C,0x02,0x00,value);
		PotiWrite2(card,0x0D,0x02,0x00);
		break;
		default:
		break;
		}
}

/****************************************************************************/
void WriteOffset00(pIk121 card,short axis, short value)
{
	if (value < -63) value = -63;
	if (value >  63) value = 63;
	card->axis[axis]->offset00 = value;
	WriteOff00(card->axis[axis]->baseadr,	card->axis[axis]->ucaxis,
				 card->axis[axis]->offset00);
}

/****************************************************************************/
void WriteOffset90(pIk121 card, short axis, short value)
{
	if (value < -63) value = -63;
	if (value >  63) value = 63;
	card->axis[axis]->offset90 = value;
	WriteOff90(card->axis[axis]->baseadr,	card->axis[axis]->ucaxis,
				 card->axis[axis]->offset90);
}

/****************************************************************************/
void TurnPhasepoti(pIk121 card, unsigned char axis, unsigned char turns,
						unsigned char updown)
{
		switch(axis)
		{
		case(1):
		if (updown) PotiInc(card,0,turns); else PotiDec(card,0,turns);
		break; 
		case(2):
		if (updown) PotiInc(card,3,turns); else PotiDec(card,3,turns);
		break;
		default:
		break;
		}
}

/****************************************************************************/
void TurnSympoti(pIk121 card, unsigned char axis, unsigned char turns,
						unsigned char updown)
{
		switch(axis)
		{
		case(1):
		if (updown) PotiInc(card,1,turns); else PotiDec(card,1,turns);
		break; 
		case(2):
		if (updown) PotiInc(card,2,turns); else PotiDec(card,2,turns);
		break;
		default:
		break;
		}
}

/****************************************************************************/
void TurnOffsetdg00(pIk121 card, unsigned char axis, unsigned char turns,
							unsigned char updown)
{
	for (unsigned char i=1; i<=turns; i++)
		{
		if (updown)
			{
			if (card->axis[axis]->offset00 < 63)
				card->axis[axis]->offset00 = card->axis[axis]->offset00+1;
			}
		else
			{
			if (card->axis[axis]->offset00 > -63)
				card->axis[axis]->offset00 = card->axis[axis]->offset00-1;
			}
		}
	WriteOff00(card->axis[axis]->baseadr, card->axis[axis]->ucaxis,
				 card->axis[axis]->offset00);
}

/****************************************************************************/
void TurnOffsetdg90(pIk121 card, unsigned char axis, unsigned char turns,
							unsigned char updown)
{
	for (unsigned char i=1; i<=turns; i++)
		{
		if (updown)
			{
			if (card->axis[axis]->offset90 < 63)
				card->axis[axis]->offset90 = card->axis[axis]->offset90+1;
			}
		else
			{
			if (card->axis[axis]->offset90 > -63)
				card->axis[axis]->offset90 = card->axis[axis]->offset90-1;
			}
		}
	WriteOff90(card->axis[axis]->baseadr,	card->axis[axis]->ucaxis,
				 card->axis[axis]->offset90);
}

/****************************************************************************/
void StorePotis(pIk121 card)
{
	// Global write to all wiper registers in register 0
	PotiWrite2(card,0x08,0x00,0x00);
}

/****************************************************************************/
/**		Default Values for all Pot Registers                              **/
/****************************************************************************/
void PotiDefault(pIk121 card)
{
	// Poti in neutral position 

	WritePhasepoti(card,1,32);
	WritePhasepoti(card,2,32);
	WriteSympoti(card,1,32);
	WriteSympoti(card,2,32);

	// Reset register 1,2,3 of pots 0,1,2,3 **/

	PotiWrite3(card,0x0C,0x00,0x01,0x00);
	PotiWrite3(card,0x0C,0x00,0x02,0x00);
	PotiWrite3(card,0x0C,0x00,0x03,0x00);
	PotiWrite3(card,0x0C,0x01,0x01,0x00);
	PotiWrite3(card,0x0C,0x01,0x02,0x00);
	PotiWrite3(card,0x0C,0x01,0x03,0x00);
	PotiWrite3(card,0x0C,0x02,0x01,0x00);
	PotiWrite3(card,0x0C,0x02,0x02,0x00);
	PotiWrite3(card,0x0C,0x02,0x03,0x00);
	PotiWrite3(card,0x0C,0x03,0x01,0x00);
	PotiWrite3(card,0x0C,0x03,0x02,0x00);
	PotiWrite3(card,0x0C,0x03,0x03,0x00);
}

/****************************************************************************/
/**		Offset is only stored in EEPROM Registers                         **/
/****************************************************************************/

void LoadOffset(pIk121 card)			// Read EEPROM 
{
	unsigned char off00_1,off90_1,off00_2,off90_2;
	
	Read4Byte(card,&off00_1,&off90_1,&off00_2,&off90_2);
	
	card->axis[1]->offset00 = off00_1;
	if (off00_1>128)	card->axis[1]->offset00 |= 0xff00;
	card->axis[1]->offset90 = off90_1;
	if (off90_1>128)	card->axis[1]->offset90 |= 0xff00;
	card->axis[2]->offset00 = off00_2;
	if (off00_2>128)	card->axis[2]->offset00 |= 0xff00;
	card->axis[2]->offset90 = off90_2;
	if (off90_2>128)	card->axis[2]->offset90 |= 0xff00;
}  

/****************************************************************************/
void Read4Byte(pIk121 card, unsigned char* off0_1, unsigned char* off9_1, 
					unsigned char* off0_2, unsigned char* off9_2)
{
	unsigned char a;
	// Load offset from register 2
	a = PotiRead(card,0x0B,0x00,0x02);
	*off0_1 = a & 0x0f;
	a = PotiRead(card,0x0B,0x00,0x03);
	a &= 0x0F;
	*off0_1 += (a <<4);

	a = PotiRead(card,0x0B,0x01,0x02);
	*off9_1 = a & 0x0f;
	a = PotiRead(card,0x0B,0x01,0x03);
	a &= 0x0F;
	*off9_1 += (a <<4);

	a = PotiRead(card,0x0B,0x02,0x02);
	*off0_2 = a & 0x0f;
	a = PotiRead(card,0x0B,0x02,0x03);
	a &= 0x0F;
	*off0_2 += (a <<4);

	a = PotiRead(card,0x0B,0x03,0x02);
	*off9_2 = a & 0x0f;
	a = PotiRead(card,0x0B,0x03,0x03);
	a &= 0x0F;
	*off9_2 += (a <<4);
}

/****************************************************************************/
void StoreOffset(pIk121 card)		// Write EEPROM 
{
	unsigned char a,b,c,d;
	a = (card->axis[1]->offset00);
	b = (card->axis[1]->offset90);
	c = (card->axis[2]->offset00);
	d = (card->axis[2]->offset90);
	Write4Byte(card,a,b,c,d);
}

/****************************************************************************/
void Write4Byte(pIk121 card, unsigned char off0_1, unsigned char off9_1,
					unsigned char off0_2, unsigned char off9_2)
{
	unsigned char a;
  // Offset wird in Register 2 gespeichert 
  a = off0_1 & 0x0F;
  PotiWrite3(card,0x0C,0x00,0x02,a);
  a = (off0_1 >>4) & 0x0f;
  PotiWrite3(card,0x0C,0x00,0x03,a);

  a = off9_1 & 0x0F;
  PotiWrite3(card,0x0C,0x01,0x02,a);
  a = (off9_1 >>4) & 0x0f;
  PotiWrite3(card,0x0C,0x01,0x03,a);

  a = off0_2 & 0x0F;
  PotiWrite3(card,0x0C,0x02,0x02,a);
  a = (off0_2 >>4) & 0x0f;
  PotiWrite3(card,0x0C,0x02,0x03,a);

  a = off9_2 & 0x0F;
  PotiWrite3(card,0x0C,0x03,0x02,a);
  a = (off9_2 >>4) & 0x0f;
  PotiWrite3(card,0x0C,0x03,0x03,a);

}

/****************************************************************************/
/**		IIC EEPROM Functions                                             **/
/****************************************************************************/
void PollRom(pIk121 card)
{
	unsigned char bit;
	unsigned char timeout = 0;
	
	do
		{
		StartIic(card);
		Out_1_Iic(card);
		Out_0_Iic(card);
		Out_1_Iic(card);
		Out_0_Iic(card);
		Out_0_Iic(card);		// A3
		Out_0_Iic(card);		// A2
		Out_0_Iic(card);   	// A1 
		Out_0_Iic(card);   	// Write 
		bit = InIic(card);
		StopIic(card);
		timeout += 1;
		}
	while (bit && (timeout < 100));
	iic_error = bit;
}

/****************************************************************************/
void RomWrite(pIk121 card, unsigned short adr,unsigned char data)
{
	PollRom(card);

	StartIic(card);
	Out_1_Iic(card);
	Out_0_Iic(card);
	Out_1_Iic(card);
	Out_0_Iic(card);
	Out_0_Iic(card);		// A2 
	Out_0_Iic(card);   	// A1
	if (adr & 0x0100) Out_1_Iic(card); else Out_0_Iic(card);
	Out_0_Iic(card);   	// Write 
	iic_error |= InIic(card);
	if (!iic_error)
		{
		if (adr & 0x80) Out_1_Iic(card); else Out_0_Iic(card);
		if (adr & 0x40) Out_1_Iic(card); else Out_0_Iic(card);
		if (adr & 0x20) Out_1_Iic(card); else Out_0_Iic(card);
		if (adr & 0x10) Out_1_Iic(card); else Out_0_Iic(card);
		if (adr & 0x08) Out_1_Iic(card); else Out_0_Iic(card);
		if (adr & 0x04) Out_1_Iic(card); else Out_0_Iic(card);
		if (adr & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (adr & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		iic_error |= InIic(card);
		if (!iic_error)
			{
			if (data & 0x80) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x40) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x20) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x10) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x08) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x04) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
			iic_error |= InIic(card);
			}
		}
	StopIic(card);          
}

/****************************************************************************/
unsigned char RomRead(pIk121 card, unsigned short adr)
{

	unsigned char data = 0;

	PollRom(card);

	StartIic(card);
	Out_1_Iic(card);
	Out_0_Iic(card);
	Out_1_Iic(card);
	Out_0_Iic(card);
	Out_0_Iic(card);   	// A2 
	Out_0_Iic(card);   	// A1 
	if (adr & 0x0100) Out_1_Iic(card); else Out_0_Iic(card);
	Out_0_Iic(card);   	// Write 
	iic_error |= InIic(card);
	if (!iic_error)
		{
		if (adr & 0x80) Out_1_Iic(card); else Out_0_Iic(card);
		if (adr & 0x40) Out_1_Iic(card); else Out_0_Iic(card);
		if (adr & 0x20) Out_1_Iic(card); else Out_0_Iic(card);
		if (adr & 0x10) Out_1_Iic(card); else Out_0_Iic(card);
		if (adr & 0x08) Out_1_Iic(card); else Out_0_Iic(card);
		if (adr & 0x04) Out_1_Iic(card); else Out_0_Iic(card);
		if (adr & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (adr & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		iic_error |= InIic(card);
		if (!iic_error)
			{
			StartIic(card);
			Out_1_Iic(card);
			Out_0_Iic(card);
			Out_1_Iic(card);
			Out_0_Iic(card);
			Out_0_Iic(card);   	// A2 
			Out_0_Iic(card);   	// A1 
			Out_0_Iic(card);   	// A0
			Out_1_Iic(card);   	// Read
			iic_error |= InIic(card);
			if (!iic_error)
				{
				if (InIic(card)) data |= 0x80;
				if (InIic(card)) data |= 0x40;
				if (InIic(card)) data |= 0x20;
				if (InIic(card)) data |= 0x10;
				if (InIic(card)) data |= 0x08;
				if (InIic(card)) data |= 0x04;
				if (InIic(card)) data |= 0x02;
				if (InIic(card)) data |= 0x01;
				Out_1_Iic(card);		// No acknowledge
				}
			}
		}
	StopIic(card);
	return(data);
}

