Program Measure;
{-------------------------------------------------------------------------}
{ Demonstration program to store a specified number of data points at a   }
{ specified time interval in milliseconds. The data is stored in an array }
{ which is then written to a file at the end of the measurment.           }
{                                                                         }
{ The program is called from the command line with the following syntax:  }
{                                                                         }
{ MEASURE  <DATAFILE> <NOOFPOINTS> <INTERVAL in ms>                       }
{         [ <PERIOD X1>[<PERIOD X2>] ]                                    }
{                                                                         }
{ Number of points has been limited to 5 000                              }
{ Data from both axes are latched simultaneously using the write_strobe   }
{ function and the interval counter to generate an internal latch signal  }
{ to both counters simultaneously. This is set up in the                  }
{ STARTDATAAQUISITION routine in the TIMER unit.                          }
{-------------------------------------------------------------------------}

Uses timer,ik120,dos,crt;

Const
    MinInterval = 1.0;              { Smallest interval in miiliseconds    }

var DataFile   : String;
    NoOfPoints : longint;
    interval   : real;              { Interval in milliseconds between pts }
    period     : array [0..1] of real; { Signal period in mm               }
    Subdivision: integer;           { Subdivision                          }
    Data: text;
    ch: char;

Procedure InitialiseVariables;
begin
     DataFile:= 'DATA.DAT';        { Set standard values }
     NoOfPoints := LocalMPMax;
     Interval:= 10;
     Period[0]:= 0.010;
     Period[1]:= 0.010;
     Subdivision:= 200;
end;

Procedure SetPeriod(ParaNum: byte; var period:real; var ErrorCode: integer );
{ Checks command line parameter x and returns value in mm }
{ If value is entered in m it is automatically corrected }
var temp: real;
begin
     if Paramstr(ParaNum) <>'' then
     begin
          Val(ParamStr(ParaNum),temp,ErrorCode);
          if temp<>0 then Period := temp;

          { Check that user has not entered period in m }
          if Period > 1 then Period := Period/1000;
     end;
end;

Function GetParametersOkay:boolean;
var ErrorCode: integer;
    temp: real;
{---------------------------------------------------------------------}
{ Command line must have the following syntax                         }
{ Measure <DataFile> <NoOfPoints> <Interval ms> [<Period X1> [<PX2>]] }
{ Number of points is checked against maximum value set in the unit   }
{ Enter grating period in mm                                          }
{---------------------------------------------------------------------}
begin
     ErrorCode:=0; temp:= 0;
     DataFile:= ParamStr(1);
     Val(ParamStr(2),NoOfPoints,ErrorCode);
     Val(ParamStr(3),Interval,ErrorCode);
     SetPeriod(4,Period[0],ErrorCode);
     SetPeriod(5,Period[1],ErrorCode);
     If (Interval < MinInterval) and (Interval<>0) then
     begin
          writeln('ERROR : Mimimum interval is ',MinInterval:6:3,
                  ' millisecond(s).');
          writeln;
     end;
     If (NoOfPoints >= LocalMPMax) then
     begin
          writeln('ERROR : Maximum number of measuring points ',LocalMPMax);
          writeln;
     end;
     GetParametersOkay:=(ErrorCode=0) and (NoOfPoints <= LocalMPMax)
                        and (ParamCount>=3) and (Interval >=MinInterval) and
                        (Interval<>0);
end;

Procedure SaveData;
begin
    {$I-}
    Assign(Data,DataFile);
    Rewrite(Data);
    {$I+}
    if IOResult<>0 then write('ERROR. Cannot write to file ',DataFile)
    else
    begin
         write('Saving data point         to  ',DataFile);
         for i:= 0 to NoOfPoints-1 do
         begin
              GotoXY(20,WhereY); write(i+1);
              write  (Data, Period[0] * Buffer[i,0]/Subdivision:8:4);
              write  (Data, Period[1] * Buffer[i,1]/Subdivision:12:4);
              writeln(Data, Period[1] * Buffer[i,1]/Subdivision -
                            Period[0] * Buffer[i,0]/Subdivision:12:4);
         end;
         close(data);
         writeln;
    end;
end;

Procedure SetCounter;
{-----------------------------------------------------------------------}
{ Counter will be initialised, set to zero and started                  }
{-----------------------------------------------------------------------}
begin
    Init_Interface(0);
    Init_Interface(1);
    Interpolation(0,I_50);
    Interpolation(1,I_50);
    Init_Counter (0,reset_start,fourfold,normal,linear);
    Init_Counter (1,reset_start,fourfold,normal,linear);
    Reset_uas(0);
    Reset_uas(1);
    Reset_Status(0);
    Reset_Status(1);
end;

{-------------------------------------------------------------------}
{ Main Program                                                      }
{-------------------------------------------------------------------}

Begin
   ClrScr;
   Writeln;
   Writeln('MEASURE - Data aquisition for IK120 at constant time intervals');
   Writeln('');
   Writeln;

   InitialiseVariables;

   if not GetParametersOkay then
   writeln('Syntax > MEASURE <Data file><NoOfPoints><Interval in ms>'+
           '[<Period in mm>]')
   else
   begin
        SetCounter;
        SetTimerInterval(Interval);
        Writeln('To start measurement :');
        Writeln('                       - press any key or ');
        Writeln('                       - use external latch input');
        Latch_enable(0,ExternalX1X2);
        Reset_Status(0);
        repeat until keypressed or Latched(0,1) or Latched(1,1);
        if keypressed then ch := readkey;
        StartDataAquisition(NoOfPoints);
        writeln('Measurement started.');
        write('Storing point : ');

        repeat
              GotoXY(20,WhereY);
              if DataIndex <BufferEnd then write(DataIndex)
                                      else write(DataIndex-1);
        until DataIndex=BufferEnd; { Waits until measurment finished }
        RemoveTimer;
        Writeln; Writeln;
        Writeln('Measurment complete.');
        Writeln;
        SaveData;
   end;
End.