{----------------------------------------------------------------------}
{  DR JOHANNES HEIDENHAIN GmbH, Traunreut, West Germany                }
{                                                                      }
{  UNIT: IK120.pas                                                     }
{                                                                      }
{  For:                             Card address:                      }
{  2 axis and latch counter         Set to $C800 as standard           }
{  Interface cards                  Muliple cards should have          }
{  Id.No.268 136 ..                 continuous addresses (max.$FC00)   }
{                                   with increments of 0400Hex         }
{----------------------------------------------------------------------}

{This is a Turbo Pascal unit intended for inclusion in user programs.
 This unit includes procedures for the basic interface functions of
 the interface card. The unit can be included in a user program using the
 "USE" statement e.g. USE dos, graphics, IK120;.
 The unit must be compiled to disk to produce the file IK120.TPU before
 the user program can be compiled.}


unit IK120;

interface
uses
  crt;
type
   control     = (c_start,c_stop,c_reset,c_latch,c_load,reset_start,reset_stop,
                 RI_start,RI_stop,RI_reset,RI_latch,
                 RI_reset_start,RI_reset_stop,
                 RI_latch_start,Ri_latch_stop,RI_latch_reset,
                 RI_latch_reset_start,RI_latch_reset_stop,RI_load,
                 Strobe_Latch);
   eval        = (onefold,twofold,fourfold);
   direction   = (normal,inverse);
   method      = (linear,arc);
   interpol    = (I_50,I_25);
   Lcontrol    = (Internal,ExternalX1,ExternalX2,ExternalX1X2,LatchOut);
   interr      = (Int_1,Int_0);

var
   boardadr   : longint;  {  Address of the first interface card  }
   m_control  : control;
   m_eval     : eval;
   m_direction: direction;
   m_method   : method;
   m_interpol : interpol;
   m_interr   : interr;
   Reg_Stat   : array[0..9,1..2] of byte;
   offset     : integer;
   regx       : byte;
   count      : longint;


PROCEDURE Init_Interface(axis:byte);
PROCEDURE Reset_Status(axis:byte);
PROCEDURE Reset_uas(axis:byte);
FUNCTION  Read_Status(axis:byte):byte;
FUNCTION  Signal_Error(axis:byte):boolean;
FUNCTION  Read_ScanReg(axis:byte):byte;
FUNCTION  Latched(axis,latch:byte):boolean;
FUNCTION  DistanceCodedREFMarks( axis: byte; BasicSpacing,SubDiv:integer;
                      Edges:eval; CountDir:Direction;Meth:Method ): longint;
PROCEDURE Interpolation(axis:byte;m_interpol:interpol);
PROCEDURE Init_Latch(axis:byte;m_control:control;m_eval:eval;value:word);
PROCEDURE Latch_Enable(axis:byte;Lcommand: Lcontrol);
PROCEDURE Latch_Disable(axis:byte;Lcommand: Lcontrol);
PROCEDURE Write_Strobe(axis: byte);
PROCEDURE Interrupt_Enable(axis:byte;m_interr:interr);
PROCEDURE Latch_Count(axis,Register:byte);
PROCEDURE Read_Count(axis,Register:byte;var Count:longint);
PROCEDURE Soft_Count(axis,Register:byte;var Count:longint);
PROCEDURE Init_Counter(axis:byte;m_control:control;m_eval:eval;
                       m_direction:direction;m_method:method);
PROCEDURE Clear_Int(axis:byte);

Implementation

{----------------------------------------------------------}
{Compiler switches to speed up routines                    }
{$B- Short Circuit bei Bedienungs Test                     }
{$R- Range Checking off                                    }
{$S- Stack Checking off                                    }
{$V+ Strict Variable Checking on                           }
{$I- I/O Checking off                                      }
{----------------------------------------------------------}

var fig: array [0..16] of string;   { Required for Large Screen Display }
    i: integer;


PROCEDURE addr(axis:byte;var offset:integer);   { Unit-interne Prozedur }
{Sets the address offset of the selected axis}
BEGIN
  offset := $0400*(axis DIV 2) + $0010*(axis MOD 2)
END;

{----------------------------------------------------------}

PROCEDURE Init_Interface(axis:byte);
{ Initialises the axis on the interface card.
  Axes 0 and 1 must be initialised with seperate procedure calls:
                                  Init_Interface(0)
                                  Init_Interface(1) }
VAR
  dummy  : byte;
BEGIN
  {  Initialialise the interface card   }
    addr(2*(axis div 2),offset);
    mem[boardadr:offset+$20]:=$0F;   {  Eval 4 fold  }
    Reg_Stat[(axis div 2),1]:=$0F;
    mem[boardadr:offset+$30]:=$80;   {  Reset  }
    mem[boardadr:offset+$30]:=$20;   {  Stop ,mask everything  }
    Reg_Stat[(axis div 2),2]:=$20;
    mem[boardadr:offset+$40]:=$FF;   {  Latch value $FF  }
    mem[boardadr:offset+$50]:=$FF;   {  Latch value $FF  }

  {  Initialise the counter  }

  addr(axis,offset);
  mem[boardadr:offset+$000F]:=$10;   {  Initialising Register    }
  mem[boardadr:offset+$000D]:=$FF;   {  Delete Status            }
  mem[boardadr:offset+$000C]:=$03;   {  Default counting method  }
  mem[boardadr:offset+$000B]:=$06;   {  Reset and stop counter   }
  mem[boardadr:offset+$000A]:=$00;   {  Ref-Impulse inactive     }
  dummy:=mem[boardadr:offset+$0003]; {  Speicher-Reg0 enable     }
  dummy:=mem[boardadr:offset+$0007]; {  Speicher-Reg1 enable     }
END;

{----------------------------------------------------------}

PROCEDURE Reset_Status(axis:byte);
{  Resets the status byte of the selected axis.}
  BEGIN
    addr(axis,offset);
    mem[boardadr:offset+$D]:=$FF;
  END;


{----------------------------------------------------------}


FUNCTION Read_Status(axis:byte):byte;
{  Reads the status byte. Evalation of the result must be carried out
  seperately.}
  BEGIN
    addr(axis,offset);
    Read_Status:=mem[boardadr:offset+$E];
  END;


{----------------------------------------------------------------}
FUNCTION Signal_Error(axis:byte):boolean;
{ Reads the status byte and returns true when a signal amplitude }
{ or frequency error has occured.                                }
  BEGIN
    Signal_Error:=(Read_Status(axis) AND $08)=$08;
  END;


{----------------------------------------------------------------}


FUNCTION Read_ScanReg(axis:byte):byte;
{  Reads value from scan-register.}
  BEGIN
    addr(axis,offset);
    Read_ScanReg:=mem[boardadr:offset+$F];
  END;

{----------------------------------------------------------------}


FUNCTION Latched(axis,Latch:byte):boolean;
{ Returns true if the latch bit has been set. }
  BEGIN
      case Latch of
      0: Latched:= ( Read_Status(axis) and $01 ) = $01;
      1: Latched:= ( Read_Status(axis) and $02 ) = $02;
      else Latched:= false;
      end;
{      delay(1); }
  END;

{----------------------------------------------------------}


PROCEDURE Reset_uas(axis:byte);
{  Resets the error bit for frequency or amplitude errors on selected axis.}
 BEGIN
    addr(2*(axis div 2),offset);
    if not odd(axis) then
      begin
        Reg_Stat[(axis div 2),1]:=Reg_Stat[(axis div 2),1] AND $FB;
        mem[boardadr:offset+$20]:=Reg_Stat[(axis div 2),1];
        delay(10);
        Reg_Stat[(axis div 2),1]:=Reg_Stat[(axis div 2),1] OR $04;
        mem[boardadr:offset+$20]:=Reg_Stat[(axis div 2),1];
      end
     else
      begin
        Reg_Stat[(axis div 2),1]:=Reg_Stat[(axis div 2),1] AND $F7;
        mem[boardadr:offset+$20]:=Reg_Stat[(axis div 2),1];
        delay(10);
        Reg_Stat[(axis div 2),1]:=Reg_Stat[(axis div 2),1] OR $08;
        mem[boardadr:offset+$20]:=Reg_Stat[(axis div 2),1];
      end;
 END;


{----------------------------------------------------------}


PROCEDURE Interpolation(axis:byte;m_interpol:interpol);
{ Sets the specified interpolation.
  e.g. Interpolation(0,I_50) sets axis 0 to 50 fold interpolation so that 200
       counts per grating period occur.}
 BEGIN
    addr(2*(axis div 2),offset);
    if not odd(axis) then
        case m_interpol of
         I_50 : Reg_Stat[(axis div 2),1]:=Reg_Stat[(axis div 2),1] OR $01;
         I_25 : Reg_Stat[(axis div 2),1]:=Reg_Stat[(axis div 2),1] AND $FE;
        end
     else
        case m_interpol of
         I_50 : Reg_Stat[(axis div 2),1]:=Reg_Stat[(axis div 2),1] OR $02;
         I_25 : Reg_Stat[(axis div 2),1]:=Reg_Stat[(axis div 2),1] AND $FD;
        end;
   mem[boardadr:offset+$20]:=Reg_Stat[(axis div 2),1];
 END;


{----------------------------------------------------------}

PROCEDURE Init_Latch(axis:byte;m_control:control;m_eval:eval;value:word);
{  Initialises the latch counter. The axis can be 0,1, 2,3, 4,5, ..
   e.g. Init_Latch(0,start,fourfold,$0F);
                  Means: Start Counter
                         Fourfold evaluation
                         Latch value $0F (Hex)}
 VAR
   value0,value1  : byte;
 BEGIN
   Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] AND $0F;
       { Sets bits D4-D7 to 0 }
   case m_control of
        c_Start  : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR $10;
        c_Stop   : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR $20;
        c_Reset  : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR $80;
        c_Latch  : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR $70;
        c_Load   : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR $B0;
        RI_Start : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR $50;
        RI_Stop  : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR $60;
        RI_Reset : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR $A0;
        RI_Latch : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR $F0;
        RI_Load  : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR $D0;
     Strobe_Latch :Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR $E0;
   end;
   Reg_Stat[(axis div 2),1]:=Reg_Stat[(axis div 2),1] AND $CF; { fourfold }
   CASE m_eval OF
        onefold : Reg_Stat[(axis div 2),1]:=Reg_Stat[(axis div 2),1] OR  $20;
        twofold : Reg_Stat[(axis div 2),1]:=Reg_Stat[(axis div 2),1] OR  $10;
        fourfold: Reg_Stat[(axis div 2),1]:=Reg_Stat[(axis div 2),1] AND $CF;
   end;

   addr(2*(axis div 2),offset);
   value:=value-1;
   if value<1 then value:=1;
   if value>65535 then value:=65535;
   value0:=lo(value);
   value1:=hi(value);
   mem[boardadr:offset+$20]:=Reg_Stat[(axis div 2),1];
   mem[boardadr:offset+$30]:=Reg_Stat[(axis div 2),2];
   mem[boardadr:offset+$40]:=value0;
   mem[boardadr:offset+$50]:=value1;
 END;


{----------------------------------------------------------}


PROCEDURE Latch_Enable(axis:byte;Lcommand:Lcontrol);
BEGIN
   case Lcommand of
   Internal   : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR  $01;
   ExternalX1 : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR  $02;
   ExternalX2 : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR  $04;
   ExternalX1X2:Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR  $06;
   LatchOut   : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] OR  $08;
   end;
   addr(2*(axis div 2),offset);
   mem[boardadr:offset+$30]:=Reg_Stat[(axis div 2),2];
 END;

PROCEDURE Latch_Disable(axis:byte;Lcommand:Lcontrol);
BEGIN
   case Lcommand of
   Internal   : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] AND $FE;
   ExternalX1 : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] AND $FD;
   ExternalX2 : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] AND $FB;
   ExternalX1X2:Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] AND $F9;
   LatchOut   : Reg_Stat[(axis div 2),2]:=Reg_Stat[(axis div 2),2] AND $F7;
   end;
   addr(2*(axis div 2),offset);
   mem[boardadr:offset+$30]:=Reg_Stat[(axis div 2),2];
 END;


PROCEDURE Write_Strobe(axis:byte);
{--------------------------------------------------------------------}
{ Writes a Strobe signal to Interval Counter of the selected card.   }
{ Can be used to trigger simultaneous software latching of both axes }
{ by programing interval counter with Strobe_Latch.                  }
{--------------------------------------------------------------------}
 BEGIN
   addr(2*(axis div 2),offset);
   mem[boardadr:offset+$60]:=$FF;
 END;


PROCEDURE Init_Counter(axis:byte;m_control:control;m_eval:eval;
                                   m_direction:direction;m_method:method);
{------------------------------------------------------------------------}
{ Initialises the specified counter.                                     }
{ e.g. Init_Counter(0,start,fourfold,normal,linear);                     }
{      Result:  Starts counter 0 					 }
{               Each edge will be counted e.g. four counts per TTL signal}
{               Normal count direction					 }
{               Linear counting method					 }
{------------------------------------------------------------------------}
  VAR
    befehl,reg_a,reg_b  : byte;
  BEGIN
      addr(axis,offset);
      mem[boardadr:offset+$F]:=$00;
      mem[boardadr:offset+$D]:=$FF;

         CASE m_eval OF
           onefold          :   CASE m_direction OF
                                  normal:  CASE m_method OF
                                             linear : befehl:=$00;
                                             arc    : befehl:=$04;
                                           END;
                                  inverse:  CASE m_method OF
                                             linear : befehl:=$08;
                                             arc    : befehl:=$0C;
                                           END;
                                END;
           twofold         :   CASE m_direction OF
                                  normal:  CASE m_method OF
                                             linear : befehl:=$01;
                                             arc    : befehl:=$05;
                                           END;
                                  inverse:  CASE m_method OF
                                             linear : befehl:=$09;
                                             arc    : befehl:=$0D;
                                           END;
                                END;
           fourfold         :   CASE m_direction OF
                                  normal:  CASE m_method OF
                                             linear : befehl:=$03;
                                             arc    : befehl:=$07;
                                           END;
                                  inverse:  CASE m_method OF
                                             linear : befehl:=$0B;
                                             arc    : befehl:=$0F;
                                           END;
                                END;
         END;
      mem[boardadr:offset+$C]:=befehl;

      reg_b:=$00;
      reg_a:=$00;
         CASE m_control OF
           RI_start            :   reg_a:=$01;
           RI_stop             :   reg_a:=$02;
           RI_reset            :   reg_a:=$04;
           RI_reset_start      :   reg_a:=$05;
           RI_reset_stop       :   reg_a:=$06;
           RI_latch            :   reg_a:=$08;
           RI_latch_start      :   reg_a:=$09;
           RI_latch_stop       :   reg_a:=$0A;
           RI_latch_reset      :   reg_a:=$0C;
           RI_latch_reset_start:   reg_a:=$0D;
           RI_latch_reset_stop :   reg_a:=$0E;
           c_start             :   reg_b:=$01;
           c_stop              :   reg_b:=$02;
           c_reset             :   reg_b:=$04;
           reset_start         :   reg_b:=$05;
           reset_stop          :   reg_b:=$06;
         END;
      mem[boardadr:offset+$B]:=reg_b;
      mem[boardadr:offset+$A]:=reg_a;
  END;


PROCEDURE Latch_Count(axis,Register:byte);
{---------------------------------------------------------------------------}
{ Latches the specified axis. The count is stored to the specified register }
{---------------------------------------------------------------------------}
 BEGIN
   addr(axis,offset);
   mem[boardadr:offset+8+Register]:=$FF;
 END;


PROCEDURE Read_Count(axis,Register:byte;var Count:longint);
{-----------------------------------------------------------------------}
{ Reads the current value from the specified register			}
{ Latching must be carried out separately 				}
{-----------------------------------------------------------------------}
 BEGIN
   addr(axis,offset);
   Count:=meml[boardadr:offset+4*Register];
 END;


PROCEDURE Soft_Count(axis,Register:byte;var Count:longint);
{------------------------------------------------------------------------}
{ Latches and reads the current value from the specified register	 }
{------------------------------------------------------------------------}
 BEGIN
   addr(axis,offset);
   mem[boardadr:offset+8+Register]:=$FF;
   Count:=meml[boardadr:offset+4*Register];
 END;


PROCEDURE Clear_Int(axis:byte);
{------------------------------------------------------------------------}
{ Resets the Interrupt register.					 }
{ Must be included in the interrupt service routine.			 }
{------------------------------------------------------------------------}
  BEGIN
    addr(2*(axis div 2),offset);
    Mem[boardadr:offset+$70]:=$00;
    Port[$20]:=$20;
  END;



PROCEDURE Interrupt_Enable(axis:byte;m_interr:interr);
{-------------------------------------------------------------------------}
{ Enable interrupt operating mode.					  }
{  e.g. Interrupt_Enable(0,Int_Internal) means that the latch counter  	  }
{       will generate an interrupt (The interrupt jumper must be set).    }
{       Interrupt_Enable(0,Int_External) means that if the external latch }
{       input is enabled an interrupt will occur when a latch signal is   }
{       detected                                                          }
{-------------------------------------------------------------------------}
BEGIN
   addr(2*(axis div 2),offset);
   case m_interr of
    Int_0: Reg_Stat[(axis div 2),1]:=Reg_Stat[(axis div 2),1] OR $40;
    Int_1: Reg_Stat[(axis div 2),1]:=Reg_Stat[(axis div 2),1] OR $80;
   end;
   mem[boardadr:offset+$20]:=Reg_Stat[(axis div 2),1];
   port[$21] := port[$21] and $00;          { Enables all IRQ lines }
END;


Function DistanceCodedREFMarks( axis:byte; BasicSpacing,SubDiv:integer;
                      Edges:eval; CountDir:Direction;Meth:Method ): longint;
{--------------------------------------------------------------------------}
{ Returns the absolute position of the first REF mark in grating periods   }
{ The axis counter is zeroed on the first REF mark.                        }
{ Spacing between the first and second REF mark is stored in count.        }
{ Formula is used to calculate the absolute position of the first REF mark.}
{ The positon is returned in units of grating periods.                     }
{ Basic Spacing is the number of grating periods between REF marks 0-2,2-4 }
{ This is normaly 1000 for encoders with 20 m grating pitch               }
{ Procedure can be interrupted by pressing any key on the keyboard.        }
{ 0 is returned if interrupted                                             }
{--------------------------------------------------------------------------}

Function Sgn(x:longint):integer;
begin
   if x>= 0 then Sgn:= 1 else Sgn:=-1;
end;

Var A,Count: longint;
begin
    { Traverse first REF mark and set to zero }
    Count:=0;
    Init_Counter(axis,Reset_Stop,Edges,CountDir,Meth);
    Init_Counter(axis,RI_Start,Edges,CountDir,Meth);
    repeat
    until (Read_status(axis) and $04=$04) or keypressed;        { Started }
    write(chr(7));

    { Latch on second REF mark. Store separatation }
    repeat
          Init_Counter (axis,RI_Latch,Edges,CountDir,Meth);
          repeat
          until Latched(axis,0) or keypressed;  { Latched }
          write(chr(7));
          Read_Count(axis,0,count);
    until (Abs(count)>10) or keypressed;   { Check that differnt REF mark 2 }
                                           { has been traversed             }
    if keypressed then DistanceCodedREFMarks:=0
    else
    begin
       { Calculate absolute position }
       A:= 2* Abs(count) DIV SubDiv - BasicSpacing;
       DistanceCodedREFMarks := (Abs(A)-Sgn(A)-1) * BasicSpacing DIV 2
                               +(Sgn(A)-Sgn(count))*Abs(Count) DIV (2*SubDiv);
    end;
end;


{-------------- main program of unit -----------------------}
BEGIN
   for i:= 0 to 9 do begin
                          Reg_Stat[i,1]:= $00;
                          Reg_Stat[i,2]:= $00;
                     end;
   boardadr:= $C800;
END. {unit}
