UNIT IK120_1;

{-------------------------------------------------------}
{ DR. JOHANNES HEIDENHAIN GmbH, Traunreut, Germany      }
{                                                       }
{ Driver Unit for IK120                                 }
{                                                       }
{ V 1.00                                                }
{ Nov 1995                                              }
{-------------------------------------------------------}

{$N+,E+}
{$V+}
{$R+}

interface

uses ik120_0,crt;

CONST
      IRQ3  =$0B;
      IRQ4  =$0C;
      IRQ5  =$0D;
      IRQ7  =$0F;

TYPE

   auswertung = (onefold,twofold,fourfold);

   initlatchcnt  = record
                     value     : word;
                     flankenaw : auswertung;
                     counter   : (start,ristart,stop,clear,
                                  nop)
                   end;
   initiklatch   = record
                     en_l0        : boolean;
                     en_l1_ext1   : boolean;
                     en_l1_ext2   : boolean;
                   end;
   initsync      = record
                     en_ausgang : boolean;
                   end;
   initintrpt    = record
                     latch_int  : boolean;
                     latch_ext  : boolean;
                    end;

   initmain      = record
                    interpol   : (i25fach,i50fach);
                    flankenaw  : auswertung;
                    inverse    : boolean;
                    arcmode    : boolean;
                  end;
   initg6latch   = record
                     en_ref       : (l0,l1);
                     en_latch0    : boolean;
                     en_latch1    : boolean;
                   end;
   softcommand   = record
                     start    : boolean;
                     stop     : boolean;
                     clear    : boolean;
                     latch0   : boolean;
                     latch1   : boolean;
                     clrerror : boolean;
                     strobe   : boolean;
                   end;
   refcommand    = record
                     ristart : boolean;
                     ristop  : boolean;
                     riclear : boolean;
                     rilatch : boolean;
                   end;
  countstate  = record
                  latch0     : boolean;
                  latch1     : boolean;
                  stop       : boolean;
                  error      : boolean;
                  ref        : boolean;
                end;

   G6_pointr    = ^G6_record;
   G6_record    = record
                     baseadr       : word;
                     axis          : byte;
                     register0     : comp;    (* Increments *)
                     register1     : comp;
                     softcomm      : softcommand;
                     refcomm       : refcommand;
                     inimain       : initmain;
                     inilatch      : initg6latch;
                     cstatus       : countstate;
                 end;

  ik120_pointr = ^ik120_record;
  ik120_record = record
                   baseadr       : word;
                   axis          : array[1..2] of G6_pointr;
                   inilatch      : initiklatch;
                   inisync       : initsync;
                   iniint        : initintrpt;
                   inilatchcnt   : initlatchcnt;
                 end;

  storage = record
               axis  : array[1..2] of G6_record;
             end;

FUNCTION  Look_for_IK120(board:IK120_pointr):boolean; (* Test for hardware *)
PROCEDURE Init_ik120(board:IK120_pointr);
FUNCTION  Read_Adr:word;                        (* Reads  IK121.INI *)
PROCEDURE Write_Adr(address:word);              (* Writes IK121.INI *)
PROCEDURE Read_Count_Status(pointr:G6_pointr);

PROCEDURE Init_G6Handler(pointr:G6_pointr);
PROCEDURE Comm_G6Handler(pointr:G6_pointr);
PROCEDURE Init_IKHandler(pointr:IK120_pointr);
PROCEDURE Comm_IKHandler(pointr:IK120_pointr);

(* Direct hardware access procedures *)
PROCEDURE Soft_Latch0(pointr:G6_pointr);
PROCEDURE Soft_Latch1(pointr:G6_pointr);
PROCEDURE Soft_strobe(pointr:IK120_pointr);
PROCEDURE Read_Reg0(pointr:G6_pointr);
PROCEDURE Read_Reg1(pointr:G6_pointr);
PROCEDURE Poll_Reg0(pointr:G6_pointr);
PROCEDURE Poll_Reg1(pointr:G6_pointr);

(* Interrupt procedures *)
PROCEDURE En_Int(Intrpt:byte);
PROCEDURE Dis_Int(Intrpt:byte);
PROCEDURE Clear_Int;

Implementation
{----------------------------------------------------------}
PROCEDURE Soft_Latch0(pointr:G6_pointr);
  BEGIN
    write_g6(pointr^.baseadr,pointr^.axis,8,$FF);
  END;
{----------------------------------------------------------}
PROCEDURE Soft_Latch1(pointr:G6_pointr);
  BEGIN
    write_g6(pointr^.baseadr,pointr^.axis,9,$FF);
  END;
{----------------------------------------------------------}
PROCEDURE Soft_strobe(pointr:IK120_pointr);
  BEGIN
    write_stb(pointr^.baseadr);
  END;
{----------------------------------------------------------}
PROCEDURE Read_Reg0(pointr:G6_pointr);
  VAR
    value : comp;
  BEGIN
    if Latched(pointr^.baseadr,pointr^.axis,0) then
      begin
        value:=Read_Count_value32(pointr^.baseadr,pointr^.axis,0);
        pointr^.register0:=value;
      end;
  END;
{----------------------------------------------------------}
PROCEDURE Read_Reg1(pointr:G6_pointr);
  VAR
    value : comp;
  BEGIN
    if Latched(pointr^.baseadr,pointr^.axis,1) then
      begin
        value:=Read_Count_value32(pointr^.baseadr,pointr^.axis,1);
        pointr^.register1:=value;
      end;
  END;
{----------------------------------------------------------}
PROCEDURE Poll_Reg0(pointr:G6_pointr);
  VAR
    eing : char;
    ende : boolean;
  BEGIN
    ende:=false;
    repeat
      if keypressed then
        begin
          eing:=readkey;  (* Stop with Ctrl C *)
          if eing=#03 then
            begin
              ende:=true;
            end;
        end;
    until (Latched(pointr^.baseadr,pointr^.axis,0) or ende);
    Read_Reg0(pointr);
  END;
{----------------------------------------------------------}
PROCEDURE Poll_Reg1(pointr:G6_pointr);
  VAR
    eing : char;
    ende : boolean;
  BEGIN
    ende:=false;
    repeat
      if keypressed then
        begin
          eing:=readkey;  (* Stop with Ctrl C *)
          if eing=#03 then
            begin
              ende:=true;
            end;
        end;
    until (Latched(pointr^.baseadr,pointr^.axis,1) or ende);
    Read_Reg1(pointr);
  END;
{----------------------------------------------------------}

{----------------------------------------------------------}
FUNCTION IK_Crt1(pointr:IK120_pointr):word;
  VAR
    buffer   : byte;
  BEGIN
    buffer:=$00;
    if (pointr^.axis[1]^.inimain.interpol=i50fach) then
      begin
        buffer:=buffer or $01;
      end;
    if (pointr^.axis[2]^.inimain.interpol=i50fach) then
      begin
        buffer:=buffer or $02;
      end;
    buffer:=buffer or $0C;  (* reset AxisA - reset AxisB *)
    if (pointr^.inilatchcnt.flankenaw=onefold) then
      begin
        buffer:=buffer or $30;
      end;
    if (pointr^.inilatchcnt.flankenaw=twofold) then
      begin
        buffer:=buffer or $10;
      end;
    if (pointr^.inilatchcnt.flankenaw=fourfold) then
      begin
        buffer:=buffer or $00;
      end;
    if pointr^.iniint.latch_int then             (* I0 *)
      begin
        buffer:=buffer or $40;
      end;
    if pointr^.iniint.latch_ext then             (* I1 *)
      begin
        buffer:=buffer or $80;
      end;
    IK_Crt1:=buffer;
  END;
{----------------------------------------------------------}
FUNCTION IK_Crt2(pointr:IK120_pointr):word;
  VAR
    buffer   : byte;
  BEGIN
    if (pointr^.inilatch.en_l0) then
      begin
        buffer:=buffer or $01;
      end;
    if (pointr^.inilatch.en_l1_ext1) then
      begin
        buffer:=buffer or $02;
      end;
    if (pointr^.inilatch.en_l1_ext2) then
      begin
        buffer:=buffer or $04;
      end;
    if (pointr^.inisync.en_ausgang) then
      begin
        buffer:=buffer or $08;
      end;

    if (pointr^.inilatchcnt.counter=start) then
      begin
        buffer:=buffer or $10;
      end;
    if (pointr^.inilatchcnt.counter=ristart) then
      begin
        buffer:=buffer or $50;
      end;
    if (pointr^.inilatchcnt.counter=stop) then
      begin
        buffer:=buffer or $20;
      end;
    if (pointr^.inilatchcnt.counter=clear) then
      begin
        buffer:=buffer or $80;
      end;

    IK_Crt2:=buffer;
  END;
{----------------------------------------------------------}
PROCEDURE Init_IKHandler(pointr:IK120_pointr);
  VAR
    buffer      : byte;
    wert        : word;
  BEGIN

     buffer:=IK_Crt1(pointr);
     write_conf(pointr^.baseadr,0,buffer);

     buffer:=IK_Crt2(pointr);
     write_conf(pointr^.baseadr,1,buffer);

     wert:=pointr^.inilatchcnt.value;
     buffer:=wert and $00FF;
     write_conf(pointr^.baseadr,2,buffer);

     wert:=pointr^.inilatchcnt.value;
     wert:=wert shr 8;
     buffer:=wert and $00FF;
     write_conf(pointr^.baseadr,3,buffer);

  END;
{----------------------------------------------------------}

{----------------------------------------------------------}
FUNCTION G6_Main(pointr:G6_pointr):word;
  VAR
    buffer   : byte;
  BEGIN
    buffer:=$00;
    if (pointr^.inimain.flankenaw=fourfold) then
      begin
        buffer:=buffer or $03;
      end;
    if (pointr^.inimain.flankenaw=twofold) then
      begin
        buffer:=buffer or $01;
      end;
    if (pointr^.inimain.flankenaw=onefold) then
      begin
        buffer:=buffer or $00;
      end;
    if (pointr^.inimain.arcmode) then buffer:=buffer or $04;
    if (pointr^.inimain.inverse) then buffer:=buffer or $08;
    G6_Main:=buffer;
  END;
{----------------------------------------------------------}
FUNCTION G6_Latch(pointr:G6_pointr):word;
  VAR
    buffer   : byte;
  BEGIN
    buffer:=$00;
    if not(pointr^.inilatch.en_latch0) then
      begin
        buffer:=buffer or $04;
      end;
    if not(pointr^.inilatch.en_latch1) then
      begin
        buffer:=buffer or $08;
      end;
    if (pointr^.inilatch.en_ref=l1) then
      begin
        buffer:=buffer or $10;
      end;
    G6_Latch:=buffer;
  END;
{----------------------------------------------------------}
PROCEDURE Init_G6Handler(pointr:G6_pointr);
  VAR
    buffer      : byte;
    wert        : word;
  BEGIN

     buffer:=G6_Main(pointr);
     write_G6(pointr^.baseadr,pointr^.axis,12,buffer);

     buffer:=G6_Latch(pointr);
     write_G6(pointr^.baseadr,pointr^.axis,15,buffer);

  END;
{----------------------------------------------------------}

{----------------------------------------------------------}
FUNCTION G6_Soft(pointr:G6_pointr):word;
  VAR
    control_buffer : byte;
  BEGIN
      control_buffer:=$00;
      if (pointr^.softcomm.start) then
        begin
          control_buffer:=control_buffer or $01;
          pointr^.softcomm.start:=false;
        end;
      if (pointr^.softcomm.stop) then
        begin
          control_buffer:=control_buffer or $02;
          pointr^.softcomm.stop:=false;
        end;
      if (pointr^.softcomm.clear) then
        begin
          control_buffer:=control_buffer or $04;
          pointr^.softcomm.clear:=false;
        end;
      G6_Soft:=control_buffer;
   END;
{----------------------------------------------------------}
FUNCTION G6_Ref(pointr:G6_pointr):word;
  VAR
    ref_buffer : byte;
  BEGIN
    ref_buffer:=$00;
      if (pointr^.refcomm.ristart) then
        begin
          ref_buffer:=ref_buffer or $01;
          pointr^.refcomm.ristart:=false;
        end;
      if (pointr^.refcomm.ristop) then
        begin
         ref_buffer:=ref_buffer or $02;
         pointr^.refcomm.ristop:=false;
       end;
      if (pointr^.refcomm.riclear) then
        begin
          ref_buffer:=ref_buffer or $04;
          pointr^.refcomm.riclear:=false;
        end;
      if (pointr^.refcomm.rilatch) then
        begin
          ref_buffer:=ref_buffer or $08;
          pointr^.refcomm.rilatch:=false;
        end;
      G6_ref:=ref_buffer;
   END;
{----------------------------------------------------------}
PROCEDURE Comm_G6Handler(pointr:G6_pointr);
  VAR
    buffer : byte;
  BEGIN

    buffer:=G6_soft(pointr);                    (* start,stop,clear *)
    if (buffer<>0) then
      begin
        write_g6(pointr^.baseadr,pointr^.axis,11,buffer);
      end;
    buffer:=G6_ref(pointr);                     (* start,stop,clear,latch *)
    if (buffer<>0) then
      begin
        write_g6(pointr^.baseadr,pointr^.axis,10,buffer);
      end;
    if pointr^.softcomm.latch0 then             (* latch0 *)
      begin
        write_g6(pointr^.baseadr,pointr^.axis,8,$FF);
        pointr^.softcomm.latch0:=false;
      end;
    if pointr^.softcomm.latch1 then             (* latch1 *)
      begin
        write_g6(pointr^.baseadr,pointr^.axis,9,$FF);
        pointr^.softcomm.latch1:=false;
      end;
    if pointr^.softcomm.strobe then             (* l1strobe *)
      begin
        write_stb(pointr^.baseadr);
        pointr^.softcomm.strobe:=false;
      end;
   END;
{----------------------------------------------------------}
PROCEDURE Comm_IKHandler(pointr:IK120_pointr);
  VAR
    buffer : byte;
  BEGIN

    if pointr^.axis[1]^.softcomm.clrerror then           (* clrerror *)
      begin
        buffer:=IK_Crt1(pointr);
        buffer:=buffer and $FB;
        write_conf(pointr^.baseadr,0,buffer);
        buffer:=buffer or $04;
        write_conf(pointr^.baseadr,0,buffer);
        write_g6(pointr^.baseadr,1,13,$FF);
        pointr^.axis[1]^.softcomm.clrerror:=false;
      end;

    if pointr^.axis[2]^.softcomm.clrerror then           (* clrerror *)
      begin
        buffer:=IK_Crt1(pointr);
        buffer:=buffer and $F7;
        write_conf(pointr^.baseadr,0,buffer);
        buffer:=buffer or $08;
        write_conf(pointr^.baseadr,0,buffer);
        write_g6(pointr^.baseadr,2,13,$FF);
        pointr^.axis[2]^.softcomm.clrerror:=false;
      end;

   END;
{----------------------------------------------------------}

{----------------------------------------------------------}
PROCEDURE Read_Count_Status(pointr:G6_pointr);
  VAR
    buffer : byte;
  BEGIN
    buffer:=Read_G6(pointr^.baseadr,pointr^.axis,14);
    if (buffer and $01) = $01 then pointr^.cstatus.Latch0:=true
                              else pointr^.cstatus.Latch0:=false;
    if (buffer and $02) = $02 then pointr^.cstatus.Latch1:=true
                              else pointr^.cstatus.Latch1:=false;
    if (buffer and $04) = $04 then pointr^.cstatus.stop:=false
                              else pointr^.cstatus.stop:=true;
    if (buffer and $08) = $08 then pointr^.cstatus.error:=true
                              else pointr^.cstatus.error:=false;
    if (buffer and $20) = $20 then pointr^.cstatus.ref:=true
                              else pointr^.cstatus.ref:=false;
  END;
{----------------------------------------------------------}
PROCEDURE Clear_Int;
BEGIN
  port[$A0]:=$20;
  port[$20]:=$20;
END;
{----------------------------------------------------------}
PROCEDURE En_Int(Intrpt:byte);
BEGIN
  case Intrpt of
    IRQ3  : begin
              port[$21]:=port[$21] and $F7;
            end;
    IRQ4  : begin
              port[$21]:=port[$21] and $EF;
            end;
    IRQ5  : begin
              port[$21]:=port[$21] and $DF;
            end;
    IRQ7  : begin
              port[$21]:=port[$21] and $7F;
            end;
  end;
END;
{----------------------------------------------------------}
PROCEDURE Dis_Int(Intrpt:byte);
BEGIN
  case Intrpt of
    IRQ3  : begin
              port[$21]:=port[$21] or $08;
            end;
    IRQ4  : begin
              port[$21]:=port[$21] or $10;
            end;
    IRQ5  : begin
              port[$21]:=port[$21] or $20;
            end;
    IRQ7  : begin
              port[$21]:=port[$21] or $80;
            end;
  end;
END;
{----------------------------------------------------------}
FUNCTION Look_for_IK120(board:IK120_pointr):boolean;
  VAR
    buffer        : byte;
    a,b           : boolean;
  BEGIN
    write_G6(board^.axis[1]^.baseadr,1,15,$00);
    write_G6(board^.axis[2]^.baseadr,2,15,$00);
    a:=true;
    b:=true;
    buffer:=read_g6(board^.axis[1]^.baseadr,1,15);
    buffer:=buffer and $10;
    if buffer<>$00 then a:=false;
    buffer:=read_g6(board^.axis[2]^.baseadr,2,15);
    buffer:=buffer and $10;
    if buffer<>$00 then b:=false;
    look_for_ik120:=a and b;
  END;
{----------------------------------------------------------}
{   Read/write init file                                   }
{----------------------------------------------------------}
PROCEDURE Write_Adr(address:word);
VAR
  initfile        : text;
  buffer          : string;
BEGIN
  Assign(initfile,'IK120.INI');
  (*$I-*)
  Rewrite(initfile);
  (*$I+*)
  if ioresult=0 then
    begin
      buffer:='base address axis 1 & 2 :';
      writeln(initfile,buffer);
      str(address,buffer);
      writeln(initfile,buffer);
      close(initfile);
    end;
END;
{----------------------------------------------------------}
FUNCTION Read_Adr:word;
VAR
  initfile          : text;
  buffer            : string;
  nofile            : boolean;
  code              : integer;
  address           : word;
BEGIN
  nofile:=true;
  Assign(initfile,'IK120.INI');
  (*$I-*)
  Reset(initfile);
  (*$I+*)
  if ioresult=0 then
    begin
      (*$I-*)
      readln(initfile,buffer);
      readln(initfile,buffer);
      (*$I+*)
      if (ioresult=0) then
        begin
          val(buffer,address,code);
          nofile:=false;
        end;
      close(initfile);
    end;
  if nofile or (code<>0) then
    begin
      address:=$C800;
      write_adr(address);
    end;
  read_adr:=address;
END;
{----------------------------------------------------------}
PROCEDURE Init_axis(axis:g6_pointr);
  BEGIN
    axis^.baseadr:=$C800;
    axis^.register0:=0;
    axis^.register1:=0;

    axis^.inimain.interpol:=i50fach;
    axis^.inimain.flankenaw:=fourfold;
    axis^.inimain.arcmode:=false;
    axis^.inimain.inverse:=false;

    axis^.inilatch.en_ref:=l1;
    axis^.inilatch.en_latch0:=false;
    axis^.inilatch.en_latch1:=false;

    axis^.softcomm.start:=false;
    axis^.softcomm.stop:=true;
    axis^.softcomm.clear:=true;
    axis^.softcomm.latch0:=false;
    axis^.softcomm.latch1:=false;
    axis^.softcomm.clrerror:=true;
    axis^.softcomm.strobe:=false;

    axis^.refcomm.ristart:=false;
    axis^.refcomm.ristop:=false;
    axis^.refcomm.riclear:=false;
    axis^.refcomm.rilatch:=false;

    axis^.cstatus.latch0:=false;
    axis^.cstatus.latch1:=false;
    axis^.cstatus.stop:=false;
    axis^.cstatus.error:=false;
    axis^.cstatus.ref:=false;

  END;
{----------------------------------------------------------}
PROCEDURE Init_ik120(board:IK120_pointr);
  BEGIN
    init_axis(board^.axis[1]);
    init_axis(board^.axis[2]);
    board^.baseadr:=Read_Adr;
    board^.axis[1]^.baseadr:=board^.baseadr;
    board^.axis[2]^.baseadr:=board^.baseadr;
    board^.axis[1]^.axis:=1;
    board^.axis[2]^.axis:=2;
    Init_G6Handler(board^.axis[1]);
    Init_G6Handler(board^.axis[2]);
    Comm_G6Handler(board^.axis[2]);
    Comm_IKHandler(board);

    board^.inilatchcnt.value:=999;
    board^.inilatchcnt.flankenaw:=fourfold;
    board^.inilatchcnt.counter:=nop;
    board^.inilatch.en_l0:=false;
    board^.inilatch.en_l1_ext1:=false;
    board^.inilatch.en_l1_ext2:=false;
    board^.iniint.latch_int:=false;
    board^.iniint.latch_ext:=false;
    board^.inisync.en_ausgang:=false;
    Init_IKHandler(board);
    Comm_G6Handler(board^.axis[1]);

  END;
{----------------------------------------------------------}
END.
