/*-----------------------------IIC.CPP--------------------------------------*/
/*                                                                          */
/*			DR. JOHANNES HEIDENHAIN GmbH, Traunreut, Germany                   */
/*                                                                          */
/*			Functions to Adjust Pots of IK 342                                  */
/*                                                                          */
/*			V 1.00                                                             */
/*			November 1998                                                         */
/*--------------------------------------------------------------------------*/
/*			The values of the amplitude and phase shift of the encoder signals */
/*       are stored in register 0 of the EEPROM.                            */
/*			The offset values of the encoder signals are stored in             */
/*       register 2 and 3 of the EEPROM.                                    */
/*--------------------------------------------------------------------------*/

#include <conio.h>
#include "ik342_1.h"                   


/****************************************************************************/
/**		Globals                                                           **/
/****************************************************************************/
unsigned char iic_error;


/****************************************************************************/
/**		Low-Level Functions: Bit Manipulations                           **/
/****************************************************************************/
void WrIic(unsigned char scl, unsigned char sda, pIk342 card)
{
	unsigned short  buffer;

	buffer = ReadConfReg (card->axis[1]->baseadr, INTREG);
	buffer = buffer & 0x08;
	if (!scl) buffer |= 0x0001;
	if (!sda) buffer |= 0x0002;
	WriteConfReg (card->axis[1]->baseadr, INTREG, buffer);
				

//	buffer = G26Latch(card->axis[4]);          
//	if(scl) buffer |= 0x0800;
//	if(sda) buffer |= 0x8000;
//	WriteG26(card->axis[4]->baseadr,4,ENDATUMREG,buffer); 
}

/****************************************************************************/
void Scl0_Sda0(pIk342 card)
{
	 WrIic(off,off,card);
}

/****************************************************************************/
void Scl1_Sda0(pIk342 card)
{
	 WrIic(on,off,card);
}

/****************************************************************************/
void Scl0_Sda1(pIk342 card)
{
	 WrIic(off,on,card);
}

/****************************************************************************/
void Scl1_Sda1(pIk342 card)
{
	 WrIic(on,on,card);
}

/****************************************************************************/
void delay_us(unsigned short us)
	{
	int l,h;
	long summ,c,d,e,max;
	max=us*2380L/1000l;
	summ=0L;
	outp(67,0);
	l=inp(64);
	h=inp(64); 
	d=l+(h<<8);
	while(summ<max)
		{
		outp(67,0);
		l=inp(64);
		h=inp(64); 
		c=l+(h<<8);
		e=d-c;
		if(e<0L)e=e+65535L;		// overflow
		summ=summ+e;
		d=c;
		}
	 }

/****************************************************************************/
void StartIic(pIk342 card)
{
	Scl1_Sda1(card);
	delay_us(10);
	Scl1_Sda0(card);
	delay_us(10);
	Scl0_Sda0(card);
	delay_us(10);
}

/****************************************************************************/
void StopIic(pIk342 card)
{
	Scl0_Sda0(card);
	delay_us(10);
	Scl1_Sda0(card);
	delay_us(10);
	Scl1_Sda1(card);
	delay_us(10000);
}

/****************************************************************************/
unsigned char InIic(pIk342 card)
{
	unsigned short buffer;
	Scl0_Sda1(card);
	delay_us(10);
	Scl1_Sda1(card);
	delay_us(10);

	buffer = ReadConfReg(card->axis[1]->baseadr, INTREG);
	buffer &= 0x04;

//	buffer = ReadG26(card->axis[4]->baseadr,4,STATUSREG12);      // Read SO 
//	buffer &= 0x0020;

	Scl1_Sda1(card);
	delay_us(10);
	Scl0_Sda1(card);
	delay_us(10);
	if (buffer == 0x04) 
		return(on);
	else
		return(off);
}

/****************************************************************************/
void Out_0_Iic(pIk342 card)
{
  Scl0_Sda0(card);
  delay_us(10);
  Scl1_Sda0(card);
  delay_us(10);
  Scl0_Sda0(card);
  delay_us(10);
}

/****************************************************************************/
void Out_1_Iic(pIk342 card)
{
  Scl0_Sda1(card);
  delay_us(10);
  Scl1_Sda1(card);
  delay_us(10);
  Scl0_Sda1(card);
  delay_us(10);
}

/****************************************************************************/
/**		Medium-Level Functions: Byte Manipulation                        **/
/****************************************************************************/
void PollPoti(pIk342 card, unsigned char adr)
{
	unsigned char  bit;
	unsigned short timeout = 0;
	
	do
		{
		StartIic(card);
		Out_0_Iic(card);
		Out_1_Iic(card);
		Out_0_Iic(card);
		Out_1_Iic(card);
		if (adr & 0x08) Out_1_Iic(card);	else Out_0_Iic(card);	// A3
		if (adr & 0x04) Out_1_Iic(card); else Out_0_Iic(card);	// A2
		if (adr & 0x02) Out_1_Iic(card); else Out_0_Iic(card);	// A1
		if (adr & 0x01) Out_1_Iic(card); else Out_0_Iic(card);	// A0
		bit = InIic(card);
		if (bit) StopIic(card);
		timeout += 1;
		}
	while ((bit) && (timeout < 100));
	iic_error = bit;
}

/*****************************************************************************/
void PotiWrite2(pIk342 card, unsigned char adr, unsigned char instruct,
									  unsigned char poti, unsigned char reg)
{
	PollPoti(card,adr);
	if (!iic_error)
		{
		if (instruct & 0x08) Out_1_Iic(card);	else Out_0_Iic(card);
		if (instruct & 0x04) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		if (reg & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (reg & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		iic_error |= InIic(card);
		}
  StopIic(card);
}

/*****************************************************************************/
void PotiWrite3(pIk342 card, unsigned char adr, unsigned char instruct,
					unsigned char poti,	unsigned char reg, unsigned char data)
{
	PollPoti(card,adr);
	
	if (!iic_error)
		{
		if (instruct & 0x08) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x04) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		if (reg & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (reg & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		iic_error |= InIic(card);
		if (!iic_error)
			{
			Out_0_Iic(card);
			Out_0_Iic(card);
			if (data & 0x20) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x10) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x08) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x04) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
			if (data & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
			iic_error |= InIic(card);
			}
		}
	StopIic(card);
}

/*****************************************************************************/
void PotiInc(pIk342 card, unsigned char adr, unsigned char poti,
								  unsigned char turns)
{
	PollPoti(card,adr);
	if (!iic_error)
		{
		Out_0_Iic(card);    		// Decrement 
		Out_0_Iic(card);
		Out_1_Iic(card);
		Out_0_Iic(card);
		if (poti & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		Out_0_Iic(card);
		Out_0_Iic(card);
		iic_error |= InIic(card);
		if (!iic_error)
			{
			if (turns > 8) turns = 8;
			for (short i=1; i<=turns; i++)
				Out_1_Iic(card);
			}
		}
	StopIic(card);
}

/*****************************************************************************/
void PotiDec(pIk342 card, unsigned char adr, unsigned char poti,
								  unsigned char turns)
{
	PollPoti(card,adr);
	if (!iic_error)
		{
		Out_0_Iic(card);    		// Increment/Decrement
		Out_0_Iic(card);
		Out_1_Iic(card);
		Out_0_Iic(card);
		if (poti & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		Out_0_Iic(card);
		Out_0_Iic(card);
		iic_error |= InIic(card);
		if (!iic_error)
			{
			if (turns > 8) turns = 8;
			for (short i=1; i<=turns; i++)
				Out_0_Iic(card);
			}
		}
	StopIic(card);
}

/****************************************************************************/
unsigned char PotiRead(pIk342 card, unsigned char adr, unsigned char instruct,
												unsigned char poti, unsigned char reg)
{
	unsigned char inbit, data = 0;
	
	PollPoti(card,adr);
	if (!iic_error)
		{
		if (instruct & 0x08) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x04) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (instruct & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (poti & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		if (reg & 0x02) Out_1_Iic(card); else Out_0_Iic(card);
		if (reg & 0x01) Out_1_Iic(card); else Out_0_Iic(card);
		iic_error |= InIic(card);
		if (!iic_error)
			{
			inbit = InIic(card);
			if (inbit) data |= 0x80;
			inbit = InIic(card);
			if (inbit) data |= 0x40;
			inbit = InIic(card);
			if (inbit) data |= 0x20;
			inbit = InIic(card);
			if (inbit) data |= 0x10;
			inbit = InIic(card);
			if (inbit) data |= 0x08;
			inbit = InIic(card);
			if (inbit) data |= 0x04;
			inbit = InIic(card);
			if (inbit) data |= 0x02;
			inbit = InIic(card);
			if (inbit) data |= 0x01;
			Out_1_Iic(card);			      // No acknowledge
			}	
		}
 StopIic(card);
 return(data);
}

/****************************************************************************/
/**		High-Level Functions                                             **/
/****************************************************************************/
unsigned char ReadPhasepoti(pIk342 card, unsigned char axis)
{
	unsigned char value;

	switch(axis)
		{
		case(1):
		value = PotiRead(card,0x00,0x09,0x00,0x00);		// Read register 0 pot 0
		break; 
		case(2):
		value = PotiRead(card,0x00,0x09,0x03,0x00);		// Read register 0 pot 1
		break;
		case(3):
		value = PotiRead(card,0x01,0x09,0x00,0x00);		// Read register 0 pot 0
		break;
		case(4):
		value = PotiRead(card,0x01,0x09,0x03,0x00);		// Read register 0 pot 1
		break;                    
		default:
		break;
		}
	return(value);
}
/****************************************************************************/
unsigned char ReadSympoti(pIk342 card, unsigned char axis)
{
	unsigned char value;
	
		switch(axis)
		{
		case(1):		// Read device0,register0,pot2
		value = PotiRead(card,0x00,0x09,0x01,0x00);		
		break; 
		case(2):		// Read device0,register0,pot3
		value = PotiRead(card,0x00,0x09,0x02,0x00);		
		break;
		case(3):		// Read device1,register0,pot2
		value = PotiRead(card,0x01,0x09,0x01,0x00);		
		break;                                                          
		case(4):		// Read device1,register0,pot3
		value = PotiRead(card,0x01,0x09,0x02,0x00);		
		break;                    
		default:
		break;
		}
	return(value);
}

/****************************************************************************/
void WritePhasepoti(pIk342 card, unsigned char axis, unsigned char value)
{
	switch(axis)
		{
		case(1):		// Write data device0,register0,pot0
		PotiWrite3(card,0x00,0x0C,0x00,0x00,value);
		PotiWrite2(card,0x00,0x0D,0x00,0x00);
		break; 
		case(2):	  	// Write data device0,register0,pot1
		PotiWrite3(card,0x00,0x0C,0x03,0x00,value);
		PotiWrite2(card,0x00,0x0D,0x03,0x00);
		break;
		case(3):		// Write data device1,register0,pot0
		PotiWrite3(card,0x01,0x0C,0x00,0x00,value);
		PotiWrite2(card,0x01,0x0D,0x00,0x00);
		break;                                      
		case(4):	  	// Write data device1,register0,pot1
		PotiWrite3(card,0x01,0x0C,0x03,0x00,value);
		PotiWrite2(card,0x01,0x0D,0x03,0x00);
		break;              
		default:
		break;
		}
}

/****************************************************************************/
void WriteSympoti(pIk342 card, unsigned char axis, unsigned char value)
{
	switch(axis)
		{
		case(1):		// Write data device0,register0,pot2
		PotiWrite3(card,0x00,0x0C,0x01,0x00,value);
		PotiWrite2(card,0x00,0x0D,0x01,0x00);
		break; 
		case(2):	  	// Write data device0,register0,pot3
		PotiWrite3(card,0x00,0x0C,0x02,0x00,value);
		PotiWrite2(card,0x00,0x0D,0x02,0x00);
		break;
		case(3):		// Write data device1,register0,pot2
		PotiWrite3(card,0x01,0x0C,0x01,0x00,value);
		PotiWrite2(card,0x01,0x0D,0x01,0x00);
		break;                                         
		case(4):	  	// Write data device1,register0,pot3
		PotiWrite3(card,0x01,0x0C,0x02,0x00,value);
		PotiWrite2(card,0x01,0x0D,0x02,0x00);
		break;
		default:
		break;
		}
}

/****************************************************************************/
void WriteOffset00(pIk342 card,short axis, short value)
{
	if (value < -63) value = -63;
	if (value >  63) value = 63;
	card->axis[axis]->offset00 = value;
	WriteOff00(card->axis[axis]->baseadr,	card->axis[axis]->ucaxis,
				 card->axis[axis]->offset00);
}

/****************************************************************************/
void WriteOffset90(pIk342 card, short axis, short value)
{
	if (value < -63) value = -63;
	if (value >  63) value = 63;
	card->axis[axis]->offset90 = value;
	WriteOff90(card->axis[axis]->baseadr,	card->axis[axis]->ucaxis,
				 card->axis[axis]->offset90);
}

/****************************************************************************/
void TurnPhasepoti(pIk342 card, unsigned char axis, unsigned char turns,
						unsigned char updown)
{
		switch(axis)
		{
		case(1):
		if (updown) PotiInc(card,0x00,0x00,turns); else PotiDec(card,0x00,0x00,turns);
		break; 
		case(2):
		if (updown) PotiInc(card,0x00,0x03,turns); else PotiDec(card,0x00,0x03,turns);
		break;
		case(3):
		if (updown) PotiInc(card,0x01,0x00,turns); else PotiDec(card,0x01,0x00,turns);
		break;
		case(4):
		if (updown) PotiInc(card,0x01,0x03,turns); else PotiDec(card,0x01,0x03,turns);
		break;                                                           
		default:
		break;
		}
}

/****************************************************************************/
void TurnSympoti(pIk342 card, unsigned char axis, unsigned char turns,
						unsigned char updown)
{
		switch(axis)
		{
		case(1):
		if (updown) PotiInc(card,0x00,0x01,turns); else PotiDec(card,0x00,0x01,turns);
		break; 
		case(2):
		if (updown) PotiInc(card,0x00,0x02,turns); else PotiDec(card,0x00,0x02,turns);
		break;
		case(3):
		if (updown) PotiInc(card,0x01,0x01,turns); else PotiDec(card,0x01,0x01,turns);
		break;
		case(4):
		if (updown) PotiInc(card,0x01,0x02,turns); else PotiDec(card,0x01,0x02,turns);
		break;
		default:
		break;
		}
}

/****************************************************************************/
void TurnOffsetdg00(pIk342 card, unsigned char axis, unsigned char turns,
							unsigned char updown)
{
	for (unsigned char i=1; i<=turns; i++)
		{
		if (updown)
			{
			if (card->axis[axis]->offset00 < 63)
				card->axis[axis]->offset00 = card->axis[axis]->offset00+1;
			}
		else
			{
			if (card->axis[axis]->offset00 > -63)
				card->axis[axis]->offset00 = card->axis[axis]->offset00-1;
			}
		}
	WriteOff00(card->axis[axis]->baseadr, card->axis[axis]->ucaxis,
				 card->axis[axis]->offset00);
}

/****************************************************************************/
void TurnOffsetdg90(pIk342 card, unsigned char axis, unsigned char turns,
							unsigned char updown)
{
	for (unsigned char i=1; i<=turns; i++)
		{
		if (updown)
			{
			if (card->axis[axis]->offset90 < 63)
				card->axis[axis]->offset90 = card->axis[axis]->offset90+1;
			}
		else
			{
			if (card->axis[axis]->offset90 > -63)
				card->axis[axis]->offset90 = card->axis[axis]->offset90-1;
			}
		}
	WriteOff90(card->axis[axis]->baseadr,	card->axis[axis]->ucaxis,
				 card->axis[axis]->offset90);
}

/****************************************************************************/
void StorePotis(pIk342 card)
{
	// Global write to all wiper registers in register 0
	PotiWrite2(card,0x00,0x08,0x00,0x00);
	PotiWrite2(card,0x01,0x08,0x00,0x00);
}                                       

/****************************************************************************/
/**		Default Values for all Pot Registers                              **/
/****************************************************************************/
void PotiDefault(pIk342 card)
{
	// Poti in neutral position 

	WritePhasepoti(card,1,32);
	WritePhasepoti(card,2,32);
	WritePhasepoti(card,3,32);
	WritePhasepoti(card,4,32);
	WriteSympoti(card,1,32);
	WriteSympoti(card,2,32);
	WriteSympoti(card,3,32);
	WriteSympoti(card,4,32);

	// Reset register 1,2,3 of pots 0,1,2,3  from device0 and device1

	PotiWrite3(card,0x00,0x0C,0x00,0x01,0x00);
	PotiWrite3(card,0x00,0x0C,0x00,0x02,0x00);
	PotiWrite3(card,0x00,0x0C,0x00,0x03,0x00);
	PotiWrite3(card,0x00,0x0C,0x01,0x01,0x00);
	PotiWrite3(card,0x00,0x0C,0x01,0x02,0x00);
	PotiWrite3(card,0x00,0x0C,0x01,0x03,0x00);
	PotiWrite3(card,0x00,0x0C,0x02,0x01,0x00);
	PotiWrite3(card,0x00,0x0C,0x02,0x02,0x00);
	PotiWrite3(card,0x00,0x0C,0x02,0x03,0x00);
	PotiWrite3(card,0x00,0x0C,0x03,0x01,0x00);
	PotiWrite3(card,0x00,0x0C,0x03,0x02,0x00);
	PotiWrite3(card,0x00,0x0C,0x03,0x03,0x00);
	PotiWrite3(card,0x01,0x0C,0x00,0x01,0x00);
	PotiWrite3(card,0x01,0x0C,0x00,0x02,0x00);
	PotiWrite3(card,0x01,0x0C,0x00,0x03,0x00);
	PotiWrite3(card,0x01,0x0C,0x01,0x01,0x00);
	PotiWrite3(card,0x01,0x0C,0x01,0x02,0x00);
	PotiWrite3(card,0x01,0x0C,0x01,0x03,0x00);
	PotiWrite3(card,0x01,0x0C,0x02,0x01,0x00);
	PotiWrite3(card,0x01,0x0C,0x02,0x02,0x00);
	PotiWrite3(card,0x01,0x0C,0x02,0x03,0x00);
	PotiWrite3(card,0x01,0x0C,0x03,0x01,0x00);
	PotiWrite3(card,0x01,0x0C,0x03,0x02,0x00);
	PotiWrite3(card,0x01,0x0C,0x03,0x03,0x00);
}

/****************************************************************************/
/**		Offset is only stored in EEPROM Registers                         **/
/****************************************************************************/

void LoadOffset(pIk342 card)			// Read EEPROM 
{
	unsigned char off00_1,off90_1,off00_2,off90_2,
					  off00_3,off90_3,off00_4,off90_4;
	
	Read8Byte(card,&off00_1,&off90_1,&off00_2,&off90_2,
						&off00_3,&off90_3,&off00_4,&off90_4);
	
	card->axis[1]->offset00 = off00_1;
	if (off00_1>128)	card->axis[1]->offset00 |= 0xff00;
	card->axis[1]->offset90 = off90_1;
	if (off90_1>128)	card->axis[1]->offset90 |= 0xff00;
	card->axis[2]->offset00 = off00_2;
	if (off00_2>128)	card->axis[2]->offset00 |= 0xff00;
	card->axis[2]->offset90 = off90_2;
	if (off90_2>128)	card->axis[2]->offset90 |= 0xff00;
	card->axis[3]->offset00 = off00_3;
	if (off00_3>128)	card->axis[3]->offset00 |= 0xff00;
	card->axis[3]->offset90 = off90_3;
	if (off90_3>128)	card->axis[3]->offset90 |= 0xff00;
	card->axis[4]->offset00 = off00_4;
	if (off00_4>128)	card->axis[4]->offset00 |= 0xff00;
	card->axis[4]->offset90 = off90_4;
	if (off90_4>128)	card->axis[4]->offset90 |= 0xff00;
}                                                    

/****************************************************************************/
void Read8Byte(pIk342 card, unsigned char* off0_1, unsigned char* off9_1, 
									 unsigned char* off0_2, unsigned char* off9_2,
									 unsigned char* off0_3, unsigned char* off9_3,
									 unsigned char* off0_4, unsigned char* off9_4)
{
	unsigned char a;
	// Load offset from register 2
	a = PotiRead(card,0x00,0x0B,0x00,0x02);
	*off0_1 = a & 0x0f;
	a = PotiRead(card,0x00,0x0B,0x00,0x03);
	a &= 0x0F;
	*off0_1 += (a <<4);
	a = PotiRead(card,0x00,0x0B,0x01,0x02);
	*off9_1 = a & 0x0f;
	a = PotiRead(card,0x00,0x0B,0x01,0x03);
	a &= 0x0F;
	*off9_1 += (a <<4);
	a = PotiRead(card,0x00,0x0B,0x02,0x02);
	*off0_2 = a & 0x0f;
	a = PotiRead(card,0x00,0x0B,0x02,0x03);
	a &= 0x0F;
	*off0_2 += (a <<4);
	a = PotiRead(card,0x00,0x0B,0x03,0x02);
	*off9_2 = a & 0x0f;
	a = PotiRead(card,0x00,0x0B,0x03,0x03);
	a &= 0x0F;
	*off9_2 += (a <<4);
	a = PotiRead(card,0x01,0x0B,0x00,0x02);
	*off0_3 = a & 0x0f;
	a = PotiRead(card,0x01,0x0B,0x00,0x03);
	a &= 0x0F;
	*off0_3 += (a <<4);
	a = PotiRead(card,0x01,0x0B,0x01,0x02);
	*off9_3 = a & 0x0f;
	a = PotiRead(card,0x01,0x0B,0x01,0x03);
	a &= 0x0F;
	*off9_3 += (a <<4);
	a = PotiRead(card,0x01,0x0B,0x02,0x02);
	*off0_4 = a & 0x0f;
	a = PotiRead(card,0x01,0x0B,0x02,0x03);
	a &= 0x0F;
	*off0_4 += (a <<4);
	a = PotiRead(card,0x01,0x0B,0x03,0x02);
	*off9_4 = a & 0x0f;
	a = PotiRead(card,0x01,0x0B,0x03,0x03);
	a &= 0x0F;
	*off9_4 += (a <<4);
}

/****************************************************************************/
void StoreOffset(pIk342 card)		// Write EEPROM 
{
	unsigned char X100,X190,X200,X290,X300,X390,X400,X490;
	
	X100 = (card->axis[1]->offset00);
	X190 = (card->axis[1]->offset90);
	X200 = (card->axis[2]->offset00);
	X290 = (card->axis[2]->offset90);
	X300 = (card->axis[3]->offset00);
	X390 = (card->axis[3]->offset90);
	X400 = (card->axis[4]->offset00);
	X490 = (card->axis[4]->offset90);
	Write8Byte(card,X100,X190,X200,X290,X300,X390,X400,X490);
}

/****************************************************************************/
void Write8Byte(pIk342 card, unsigned char off0_1, unsigned char off9_1,
									  unsigned char off0_2, unsigned char off9_2,
									  unsigned char off0_3, unsigned char off9_3,
									  unsigned char off0_4, unsigned char off9_4)
{                                  
	unsigned char a;
  // Offset wird in Register 2 gespeichert 
  a = off0_1 & 0x0F;
  PotiWrite3(card,0x00,0x0C,0x00,0x02,a);
  a = (off0_1 >>4) & 0x0f;
  PotiWrite3(card,0x00,0x0C,0x00,0x03,a);
  a = off9_1 & 0x0F;
  PotiWrite3(card,0x00,0x0C,0x01,0x02,a);
  a = (off9_1 >>4) & 0x0f;
  PotiWrite3(card,0x00,0x0C,0x01,0x03,a);
  a = off0_2 & 0x0F;
  PotiWrite3(card,0x00,0x0C,0x02,0x02,a);
  a = (off0_2 >>4) & 0x0f;
  PotiWrite3(card,0x00,0x0C,0x02,0x03,a);
  a = off9_2 & 0x0F;
  PotiWrite3(card,0x00,0x0C,0x03,0x02,a);
  a = (off9_2 >>4) & 0x0f;
  PotiWrite3(card,0x00,0x0C,0x03,0x03,a);
  a = off0_3 & 0x0F;
  PotiWrite3(card,0x01,0x0C,0x00,0x02,a);
  a = (off0_3 >>4) & 0x0f;
  PotiWrite3(card,0x01,0x0C,0x00,0x03,a);
  a = off9_3 & 0x0F;
  PotiWrite3(card,0x01,0x0C,0x01,0x02,a);
  a = (off9_3 >>4) & 0x0f;
  PotiWrite3(card,0x01,0x0C,0x01,0x03,a);
  a = off0_4 & 0x0F;
  PotiWrite3(card,0x01,0x0C,0x02,0x02,a);
  a = (off0_4 >>4) & 0x0f;
  PotiWrite3(card,0x01,0x0C,0x02,0x03,a);
  a = off9_4 & 0x0F;
  PotiWrite3(card,0x01,0x0C,0x03,0x02,a);
  a = (off9_4 >>4) & 0x0f;
  PotiWrite3(card,0x01,0x0C,0x03,0x03,a);
}


