{$N-$E+}
{-------------------------------------------------------------------------}
{ Dr JOHANNES HEIDENHAIN, Traunreut, West Germany                         }
{                                                                         }
{ Demonstration program to illustrate the programming of the interface    }
{ card IK110 using the IK110.PAS unit supplied.                           }
{                                                                         }
{ Carries out normal function of a simple counter.                        }
{ Written by: K. Fairley                                                  }
{ Uses external files:-  COUNTPAR.DAT  Stores setup information or        }
{                                      parameters for the counter         }
{                        COUNTPRE.DAT  File with preset values used to    }
{                                      reproduce datum points when        }
{                                      working with reference marks       }
{ This program is intended to be modified and adapted for customer        }
{ specific applications or can be used as a simple counter display with   }
{ up to 8 axes.                                                           }
{ The use of the interval counter is also demonstrated.                   }
{ The interval counter generates an internal interrupt which is used to   }
{ store data to a file.                                                   }
{ This will only function if the interrupt jumper is set to IRQ5 for      }
{ an INTERNAL latch.                                                      }
{-------------------------------------------------------------------------}

program COUNTER;

uses IK120, BigDisp,crt,dos,printer;   { External units used by program    }

Const
     Header1 = '';
     Header2 = '   HEIDENHAIN  IK 110  -    COUNTER PROGRAM   1.02 ';
     Header3 = '';

     MaxAxes= 8;                      { Depends on free slots in PC       }
     LatchAxis= 1; { Which Axes is used as latch input i.e. X2 of board 1 }

     Mask = '@@@@@@@@@@';             { Total No. of chars in measuring val }
     Int5Vect      = $0D;
     TextCol = yellow;
     BackCol = black;


var
   Period :   array [0.. MaxAxes] of real;
   REFSpacing: array[0.. MaxAxes] of integer;
   SubDivision : integer;             { Valid for all axes                }
   Axes,                              { Number of axes in use             }
   DecPlaces,                         { Number of decimal places          }
   StoreReg:    byte;
   L_string: array [0..MaxAxes] of string;    { Last chars. on the screen }
   count: array [0..MaxAxes] of longint;               { Count from IK110 }
   Preset: array [0..MaxAxes] of real;                     { Preset value }
   REFPreset: array [0..MaxAxes] of real;               { REFPreset value }
   key: char;
   data: text;
   values: text;
   REF,DifDisplay: Boolean;
   AxesNames: string;                  { Axes designation e.g. XYZ45678   }
   Min,Max,Dif: real;
   Mins,Maxs,Difs: string;
   LatchValue: word;              { Number of counts for interval counter }
   OldInt5Vect :Pointer;


Procedure InitialiseVariables;
{-------------------------------------------------------------------------}
{ Variables are initialised at the start of the program                   }
{ Counters for each axes are initialised and set to zero                  }
{-------------------------------------------------------------------------}
var i: byte;
begin
    clrscr;
    Axes := 7;
    DecPlaces:=3;
    Boardadr:=$C800;
    AxesNames:='12345678';
    StoreReg := 0;
    REF := false;
    Subdivision := 200;
    m_interpol:=I_50;
    Dif:=0;
    Min:=0;
    Max:=0;
    Difs:=Mask;
    Mins:=Mask;
    Maxs:=Mask;
    DifDisplay := false;
    LatchValue := 0;

    { Setup each axis }
    for i:= 0 to Axes do
    begin
      Period[i] := 0.010;   { Grating period in mm }
      Count[i] := 0;
      Preset[i] := 0;
      REFSpacing[i] := 0;
      REFPreset[i] := 0;
      Init_Interface(i);
      Interpolation(i,m_interpol);
      Init_Counter (i,reset_start,fourfold,normal,linear);
      Reset_uas(i);
      Reset_Status(i);
      L_string[i] := Mask;
    end;
end;


Procedure InverseVideo;
{-------------------------------------------------------------------------}
{ Sets the screen to inverse video                                        }
{-------------------------------------------------------------------------}

var t,b: byte;         { TextAttr bits 0-3 TextColour  4-7 Background }
begin
   t := TextAttr mod 16;
   b := TextAttr div 16;
   TextColor(b);
   TextBackground(t);
end;



Procedure SavePresets;
{-------------------------------------------------------------------------}
{ Preset values saved to COUNTPRE.DAT when in REF mode                    }
{-------------------------------------------------------------------------}
var i: byte;
begin
   {$I-}
   Assign( data, 'COUNTPRE.DAT');
   Rewrite(data);
   {$I+}
   if IOResult =0 then
   begin
        For i := 0 to Axes do writeln( data, Preset[i]+REFPreset[i]:8:3);
        close(data);
   end;
end;



Procedure ReadPresets;
{-------------------------------------------------------------------------}
{ Preset values read from COUNTPRE.DAT when in REF mode                   }
{-------------------------------------------------------------------------}
var i: byte;
begin
   {$I-}
   Assign( data, 'COUNTPRE.DAT');
   Reset(data);
   {$I+}
   if IOResult = 0 then
   begin
        For i := 0 to Axes do readln( data, Preset[i]);
        close(data);
   end;
end;


Procedure ReadParameters;
{-------------------------------------------------------------------------}
{ Parameters read from COUNTPRE.DAT when starting program                 }
{-------------------------------------------------------------------------}
var comment: string;
var i: byte;
begin
   writeln('Reading parameters from COUNTPAR.DAT ...');
   {$I-}
   Assign( data, 'COUNTPAR.DAT');
   Reset(data);
   {$I+}
   if IOResult=0 then
   begin
       Readln(data, comment);
       Readln(data, Axes);
       Axes:= Axes-1;
       Readln(data, comment);
       Readln(data, BoardAdr);
       Readln(data, comment);
       Readln(data, DecPlaces);
       Readln(data, comment);
       Readln(data, AxesNames);
       Readln(data, comment);
       For i := 0 to Axes do
         readln( data, Period[i],REFSpacing[i]);
       close(data);
   end;
end;



Procedure PrintAxes;
{-------------------------------------------------------------------------}
{ Display values are set to LPT1:                                         }
{ This could be modified to stored data to a file.                        }
{-------------------------------------------------------------------------}
var i: byte;
begin
    Gotoxy(1,24); ClrEol;
    write('Printing ...');
    for i:= 0 to Axes
    do writeln(Lst,copy(AxesNames,i+1,1),
               count[i]*Period[i]/Subdivision -Preset[i]:
               4+DecPlaces:DecPlaces);
    Gotoxy(1,24); ClrEol;
end;



Procedure ReadCounts;
{-------------------------------------------------------------------------}
{ Reads the values from the storage registers on the IK110. If the        }
{ interval counter is not enabled the storage resisters are latched and   }
{ hence updated with the current count value before being read.           }
{-------------------------------------------------------------------------}
var j: integer;
begin

   { Carry out Software Latch if Interval Counter not being used }
   if LatchValue=0
   then For j:= 0 to Axes do Latch_Count(j, StoreReg );

   { Read Values from storage registers on the interface card }
   For j:= 0 to Axes do Read_Count(j, StoreReg, Count[j] );

end;


Procedure DisplayCounts;
{-------------------------------------------------------------------------}
{ The values in the count array are displayed on the screen. Upto 3 axes  }
{ are displayed in large rounded figures. 4 to 8 axes are displayed in    }
{ slightly smaller figures.                                               }
{-------------------------------------------------------------------------}

var s: string;
    i: integer;
begin
     s:= '';
     for i := 0 to Axes do
     begin
        s:= '';
        if Axes<3
        then begin
                  LDisplay(Count[i]*Period[i]/Subdivision- Preset[i],Large,
                           3,i*7+5,4+DecPlaces,DecPlaces,false,
                           Signal_error(i),L_string[i]);
             end
        else begin
                  GotoXY(1+(i mod 2)*40 ,(i div 2)*5+6);
                  write(copy(AxesNames,i+1,1));
                  LDisplay(Count[i]*Period[i]/Subdivision- Preset[i],medium,
                           3+(i mod 2)*40 ,(i div 2)*5+6 ,
                           4+DecPlaces,DecPlaces,false,
                           Signal_error(i),L_string[i]);
             end;
     end;
end;

Procedure DisplaySumOrDifference;
{-------------------------------------------------------------------------}
{ Code to calculate sum, difference, min, max displays should be          }
{ inserted here.                                                          }
{                                                                         }
{ This procedure could be used for customer specific display routines.    }
{                                                                         }
{-------------------------------------------------------------------------}

var s: string;
begin
     if DifDisplay and (Axes=1) then
     begin
          Dif:= (count[1]*Period[1]/Subdivision)-Preset[1]
               - ( (count[0]*Period[0]/Subdivision)-Preset[0] );
          LDisplay(Dif,large,3,2*7+5 ,4+DecPlaces,DecPlaces,
                   false,false,Difs);
     end;

end;

Procedure DisplayInterval;
begin
   If LatchValue <>0 then
   begin
        GotoXY(70,2);ClrEol; write('Interval');
        GotoXY(70,3);ClrEol; write(LatchValue*Period[1]/Subdivision*4
                                   :4+DecPlaces:4);
   end
   else
   begin
        GotoXY(70,2);ClrEol;
        GotoXY(70,3);ClrEol;
   end;
end;

Procedure Display;
begin
     ReadCounts;
     DisplayCounts;
     DisplaySumOrDifference;
end;

Procedure ZeroAxes;
{-------------------------------------------------------------------------}
{ Axes are set to zero.                                                   }
{ In REF mode the counter zero is on the REF mark and display is zeroed   }
{ using preset value. When not operating in REF mode the counters are     }
{ zeroed on the IK110                                                     }
{-------------------------------------------------------------------------}
var i: byte;
begin
   for i := 0 to Axes do
   begin
      if REF then          { Zero remains on REF mark. Preset used for zero }
      begin
           Preset[i] := Count[i]*Period[i]/Subdivision;
           Init_Counter (i,c_start,fourfold,normal,linear);
      end
      else                 { Set counter and Preset to zero      }
      begin
           Preset[i] := 0;
           Init_Counter (i,reset_start,fourfold,normal,linear);
      end;
      Reset_uas(i);
      Reset_Status(i);
      L_string[i] := Mask;
   end;
   If REF then SavePresets;
end;







Procedure SetAxes(FirstAxis,LastAxis : byte);
{-------------------------------------------------------------------------}
{ The user is asked for a value for axes from FirstAxis to LastAxis.      }
{ The display is set to the specified value using the presets.            }
{ The new datum points or preset values are stored to COUNTPRE.DAT.       }
{-------------------------------------------------------------------------}
var x: real;
    s: string;
    dummy,i: integer;
begin
   For i := FirstAxis to LastAxis do
   begin
        s := '';
        x:= 0;
        GotoXY(1,24); ClrEol;
        write('Set Axis '+AxesNames[i+1]+ ' = ',s);
        GotoXY(WhereX-length(s),WhereY);

        readln(s);
        if s <> '' then
        begin
             val(s,x,dummy);
             if (dummy <> 0) or (abs(x)>1000)
             then write(chr(7))
             else Preset[i] :=  Count[i]*Period[i]/Subdivision - x;
        end;
        Display;
   end;
   GotoXY(1,24); ClrEol;
   if REF then SavePresets;
end;





Procedure IntServiceRoutine( Flages,CS,IP,AX,BX,CX,DX,SI,DI,DS,ES,BP:word);
{-------------------------------------------------------------------------}
{ Reads the values from the storage registers on the IK110. Writes them   }
{ and their difference to a file.                                         }
{-------------------------------------------------------------------------}
Interrupt;

var j: integer;
Begin
   ReadCounts;
   j:= 0;
   Repeat
       write( values,
              Count[j]*Period[j]/Subdivision:4+DecPlaces:DecPlaces,'    ');
       j:=j+1;
   until j> Axes;
   writeln(values,( Count[0]*Period[0]/Subdivision- Preset[0])-
                ( Count[1]*Period[1]/Subdivision- Preset[1]):
                4+DecPlaces:DecPlaces);
   Clear_Int(LatchAxis);
End;

Procedure RemoveInterrupt;
{-------------------------------------------------------------------------}
{ Interrupt vectors are reset and interupt controller disabled.           }
{-------------------------------------------------------------------------}

Begin
    Clear_Int(LatchAxis);
    SetIntvec(Int5Vect,OldInt5Vect);
    port[$21]:= port[$21] or $20;
End;

Procedure InstallInterrupt;
{-------------------------------------------------------------------------}
{ Interrupt vectors saved and set. Interupt controller disabled.          }
{-------------------------------------------------------------------------}

Begin
    GetIntvec(Int5Vect,OldInt5Vect);
    SetIntvec(Int5Vect,Addr(IntServiceRoutine) );
End;




Procedure UseIntervalCounter;
{-------------------------------------------------------------------------}
{ User is asked to enter a distance at which the interval counter latches }
{ the counters on the IK110. The value is rounded to a whole number of    }
{ grating periods, interval counter is programmed with the correct number }
{ of counts, enabled and reset.                                           }
{ When interval counter is enabled the normal software latching does not  }
{ occur and the interval counter generates latch signals.                 }
{ The interval counter is connect internaly to axis input X2 (axes 1).    }
{-------------------------------------------------------------------------}
var s,fname: string;
    x: real;
    dummy: integer;
    EXE: integer;
begin
     if M_Interpol=I_25 then EXE:= 25
                        else EXE:= 50;
     s:='';
     fname:='';
     repeat
           GotoXY(1,24);
           ClrEol;
           write('Enter interval = ');
           readln(s);
           if s='' then s:= '0';
           val(s,x,dummy);
     until (dummy=0) and (x/period[1]*EXE < 65536);
     LatchValue :=(Round(x/period[1])* EXE)
                                  MOD 65536;
     GotoXY(1,24);
     ClrEol;


     DisplayInterval;

     if LatchValue > 0 then
     begin
          repeat
                fname:= 'DATA.DAT';
                GotoXY(1,24);
                ClrEol;
                write('Enter filename for data  ['+fname+'] ');
                readln(s);
                if s<> '' then fname := s;
                {$I-}
                assign(values,fname);
                rewrite(values);
                {$I+}
          until IOResult=0;
          InstallInterrupt;
          Clear_Int(LatchAxis);
          Interrupt_Enable(0, Int_0);
          Latch_Enable(LatchAxis,Internal);
          Init_Latch(LatchAxis,C_Stop,oneFold,LatchValue);
          Init_Latch(LatchAxis,C_reset,oneFold,LatchValue);
          Init_Latch(LatchAxis,RI_start,oneFold,LatchValue);
          GotoXY(1,24);ClrEol;
          write ( 'Latching starts on REF mark X2');
     end
     else if fname <>'' then
     begin
        RemoveInterrupt;
        {$I-}
        Close(values);
        {$I+}
     end;
end;










Procedure RefAxes;
{-------------------------------------------------------------------------}
{ Counters are set to zero on the REF mark.                               }
{ Old preset values are read from COUNTPRE.DAT to reset the previous      }
{ point.                                                                  }
{ If the Axes have distance coded REF marks the counter is zeroed on      }
{ first REF mark. After second REF mark is detected the absolute position }
{ of the first REF is deteremined.                                        }
{ The position is used to modify the preset value for the axis since      }
{ the preset value refers to the difference between the REF and datum     }
{ When storing the preset value it is modified with reference to the      }
{ first REF mark.                                                         }
{-------------------------------------------------------------------------}
var temp: real;
    i: integer;
begin
   ReadPresets;
   for i:=  0 to Axes do Init_Counter(i,C_Stop,fourfold, normal, linear);
   for i := 0 to Axes do
   begin
        GotoXY(1,24);
        write( 'Traverse REF mark in axis '+AxesNames[i+1]+' ...');
        if REFSpacing[i] = 0 then
        begin
             Init_Counter (i,reset_stop,fourfold,normal,linear);
             Init_Counter (i,RI_Start,fourfold,normal,linear);
             Reset_uas(i);
             Reset_Status(i);
             L_string[i] := mask;
             repeat
                 Display;
             until ((Read_Status(i) and $04)=$04) or   { Started }
                   keypressed;
             write(chr(7));
             REF := true;
        end
        else
        begin
             REFPreset[i]:= DistanceCodedREFMarks(i,REFSpacing[i],Subdivision,
             fourfold,normal,linear) * period[i];
             Preset[i]:= Preset[i]-REFPreset[i];
             if REFPreset[i] <> 0 then REF := true
                                  else REF:= false;
        end;
   end;
   GotoXY(1,24); ClrEol;
end;






Procedure SetDifDisplay;
begin
   DifDisplay:= not DifDisplay;
   if not DifDisplay then
   begin
        Window(3,2*7+5,78,23);
        ClrScr;
        Window(1,1,80,25);
        Difs:=Mask;
   end;
end;

Procedure DrawRefMark;
begin
    Gotoxy(70,1);
    if REF then write('REF') else write('   ');
end;


Procedure Menu(key: char);
begin
     if (pos(key,AxesNames)<>0) and (pos(key,AxesNames)<=Axes+1)
     then SetAxes(pos(key,AxesNames)-1,pos(key,AxesNames)-1)

     else                           { Not XYZ etc. then check menu }
       case key of
       'N': ZeroAxes;
       'R': RefAxes;
       'S': SetAxes(0,Axes);
       'P': PrintAxes;
       'D': SetDifDisplay;
       'I': If Axes >= 1 then UseIntervalCounter;
       end;

       GotoXY(70,24); ClrEol;
end;


{-------------------------------------------------------------------------}
{                           main program                                  }
{-------------------------------------------------------------------------}

BEGIN

    InitialiseVariables;
    ReadParameters;
    ZeroAxes;
    ClrScr;
    writeln('             ',Header1);
    writeln('             ',Header2);
    writeln('             ',Header3);

    InverseVideo;
    Gotoxy(1,25);ClrEol;

    write('E)nde    N)ullen    R)ef    '+Copy(AxesNames,1,Axes+1)+' S)et    P)rint   ');
    if Axes>=1 then write('I)nterval   ');
    if Axes=1 then write('D)ifference   ');
    NormVideo;
                                                     { Display values }
    repeat
       DrawRefMark;
       repeat
             Display;
             Clear_Int(LatchAxis);
       until keypressed;

       key := Upcase(readkey);
       Menu(key);
    until key in [#27,'e','E','b','B'];

    if Latchvalue <>0 then begin
                                RemoveInterrupt;
                                {$I-}
                                close(values);
                                {$I+}
                           end;
    ClrScr;
END.