/*----------------------------IK342_0.C-------------------------------

  DR. JOHANNES HEIDENHAIN GmbH, Traunreut, Germany

  Driver Unit for IK342 (Basic Functions)

  V 1.00
  November 1998
 -------------------------------------------------------------------*/

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include "ik342_0.h"
#include "vmerotec.h"

/*----------------------Functions-----------------------------------*/
/*--------------------------------------------------------------------
                        WriteRegister
 ---------------------------------------------------------------------
 This function writes a value in a 16-bit
 register of a counter.
 -------------------------Parameters---------------------------------
 usBaseAddress       base address of the IK 342
 ucAxis	             axis select - axis 1 to axis 4
 usRegisterAddress   address of the register
 usDatum	           value to write to the register address
 usPortAddress       port address in which <usDatum> is written
 -------------------------------------------------------------------*/
void WriteRegister (unsigned short usBaseAddress, unsigned char ucAxis,
               unsigned short usRegisterAddress,
               unsigned short usDatum)
{
unsigned short usPortAddress;

switch (ucAxis)
       {
       case 1:
       usPortAddress = usBaseAddress + 0x0080 + usRegisterAddress;
       break;
       case 2:
       usPortAddress = usBaseAddress + 0x00C0 + usRegisterAddress;
       break;
       case 3:
       usPortAddress = usBaseAddress + 0x0100 + usRegisterAddress;
       break;
       case 4:
       usPortAddress = usBaseAddress + 0x0140 + usRegisterAddress;
       break;
       default:
       printf ("Wrong axis in function <WriteRegister>");
       }

   /* Write <usDatum> to the counter */
outpw (usPortAddress, usDatum);
}

/*--------------------------------------------------------------------
                       ReadRegister
 ---------------------------------------------------------------------
 This function reads a value from a 16-bit
 register of a counter.
 -----------------------Parameters------------------------------------
 usBaseAddress       basic address A0 to A9 of the IK 342
 ucAxis              axis select - axis 1 or axis 2
 usRegisterAddress   address of the register
 usPortAddress       port address in which <usDatum> is written
 -------------------------------------------------------------------*/
unsigned short ReadRegister (unsigned short usBaseAddress,
               unsigned char ucAxis, unsigned short usRegisterAddress)
{
unsigned short usPortAddress;

/* Calculate port usRegisterAddress */
switch (ucAxis)
       {
       case 1:
       usPortAddress = usBaseAddress + 0x0080 + usRegisterAddress;
       break;
       case 2:
       usPortAddress = usBaseAddress + 0x00C0 + usRegisterAddress;
       break;
       case 3:
       usPortAddress = usBaseAddress + 0x0100 + usRegisterAddress;
       break;
       case 4:
       usPortAddress = usBaseAddress + 0x0140 + usRegisterAddress;
       break;
       default:
       printf ("Wrong axis in function <WriteRegister>");
       }
   /* Read <usDatum> from the counter */
return(inpw(usPortAddress));
}
/*--------------------------------------------------------------------
                         SoftLatch_0
 ---------------------------------------------------------------------
 This function reads the measured value and stores
 it in data register 0.
 -------------------------------------------------------------------*/
void SoftLatch_0 (unsigned short usBaseAddress, unsigned char ucAxis)
{
WriteRegister (usBaseAddress, ucAxis, 0x20, 0x0001);
}


/*-------------------------------------------------------------------
                         SoftLatch_1
 ---------------------------------------------------------------------
 This function reads the measured value and stores
 it in data register 1.
 -------------------------------------------------------------------*/
void SoftLatch_1 (unsigned short usBaseAddress, unsigned char ucAxis)
{
WriteRegister (usBaseAddress, ucAxis, 0x20, 0x0002);
}

/*--------------------------------------------------------------------
                         CountValueLatched
 ---------------------------------------------------------------------
 This function checks whether a measured value is
 latched in data register 0 or 1.
 -------------------------------------------------------------------*/

unsigned char CountValueLatched (unsigned short usBaseAddress,
                  unsigned char ucAxis, unsigned char ucRegister)
{
unsigned char result;
switch (ucRegister)
       {
       case 0:
       result = (unsigned char)
                (ReadRegister (usBaseAddress, ucAxis, 0x020) & 0x0001);
       break;
       case 1:
       result = (unsigned char)
                (ReadRegister (usBaseAddress, ucAxis, 0x020) & 0x0002);
       break;
       }
return (result);
}


/*--------------------------------------------------------------------
                       PollForLatched
 ---------------------------------------------------------------------
 This function polls until a measured value is
 latched in data register 0 or 1.
 -------------------------------------------------------------------*/

void PollForLatched (unsigned short usBaseAddress,
                     unsigned char ucAxis,
                     unsigned char ucRegister)
{
switch (ucRegister)
       {
       case 0:
       while (CountValueLatched (usBaseAddress, ucAxis, 0) == 0)
       ;
       break;

       case 1:
       while (CountValueLatched (usBaseAddress, ucAxis, 1) == 0)
       ;
       break;
       }
}

/*-------------------------------------------------------------------
                    ReadCountValue_32
 --------------------------------------------------------------------
 This function reads 32 bits of a measured value.
 -------------------------------------------------------------------*/
long ReadCountValue_32 (unsigned short usBaseAddress,
                  unsigned char ucAxis, unsigned char ucRegister)
{
union 	mapper
        {
        long field0;
        unsigned short field1[2];
        }buffer;

switch (ucRegister)
       {
       case 0:
       buffer.field1[0] = ReadRegister (usBaseAddress, ucAxis, 0x3c);
       buffer.field1[1] = ReadRegister (usBaseAddress, ucAxis, 0x38);
       break;
       case 1:
       buffer.field1[0] = ReadRegister (usBaseAddress, ucAxis, 0x30);
       buffer.field1[1] = ReadRegister (usBaseAddress, ucAxis, 0x2c);
       break;
       }
return (buffer.field0);
}

/*--------------------------------------------------------------------
                    ReadCountValue_48
 ---------------------------------------------------------------------
 This function reads 48 bits of a measured value.
 -------------------------------------------------------------------*/
double ReadCountValue_48 (unsigned short usBaseAddress,
                  unsigned char ucAxis, unsigned char ucRegister)
{
unsigned short usField[3];
double count_value48;

switch (ucRegister)
       {
       case 0:
       usField[0] = ReadRegister (usBaseAddress, ucAxis, 0x3C);
       usField[1] = ReadRegister (usBaseAddress, ucAxis, 0x38);
       usField[2] = ReadRegister (usBaseAddress, ucAxis, 0x34);
       break;
       case 1:
       usField[0] = ReadRegister (usBaseAddress, ucAxis, 0x30);
       usField[1] = ReadRegister (usBaseAddress, ucAxis, 0x2C);
       usField[2] = ReadRegister (usBaseAddress, ucAxis, 0x28);
       break;
       }

if (usField[2] & 0x8000)
       count_value48 = (double)((usField[0] - 65535.0) +
       65536.0*(usField[1]-65535.0)+
       4294967296.0*(usField[2]-65535.0)-1);
       else
       count_value48 = (double)(usField[0] +
       65536.0*usField[1] +
       4294967296.0*usField[2]);


return (count_value48);
}


/*---------------------------------------------------------------------------
							  ReadConfReg
 ----------------------------------------------------------------------------
 This function reads one configuration-Register from IK342
 -----------------------Parameters-------------------------------------------
 ---------------------------------------------------------------------------*/
unsigned char ReadConfReg(unsigned short usBaseAddress,
								  unsigned short usRegisterAddress)
{
	unsigned short usPortAddress;

	usPortAddress = usBaseAddress + 0x0180 + usRegisterAddress;
	return(inp(usPortAddress+1));
}  

/*---------------------------------------------------------------------------
							  WriteConfReg
 ----------------------------------------------------------------------------
 This function writes the config-register from IK342
 -----------------------Parameters-------------------------------------------
 ---------------------------------------------------------------------------*/
void WriteConfReg(unsigned short usBaseAddress, unsigned short usRegisterAddress,
						unsigned char usDatum)
{
	unsigned short usPortAddress;

	usPortAddress = usBaseAddress + 0x0180 + usRegisterAddress;
	outp(usPortAddress+1, usDatum);		// Write <usDatum> to the IK342
} 
													  

