/*----------------------IK121_0.C-----------------------

  DR. JOHANNES HEIDENHAIN GmbH, Traunreut, Germany

  Driver Unit for IK121 (Basic Functions)

  V 1.01
  April 1995
 ------------------------------------------------------*/

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include "ik121_0.h"

/*----------------------Functions----------------------*/
/*------------------------------------------------------
                        write_g26
 -------------------------------------------------------
 This function writes a value in a 16-bit
 register of a counter.
 -------------------------Parameters--------------------
 baseadr   basic address A0 to A9 of the IK 121
 axis	   axis select - axis 1 or axis 2
 address   address of the counter B0 to B4
 datum	   value to write to the address
 adr_reg   address of the address register on the IK 121
 adr_point the part of the address indicated by the
           address register (R0 bis R2)
 adr_gate  port address in which <datum> is written
 ------------------------------------------------------*/
void write_g26 (unsigned int baseadr, unsigned char axis,
               unsigned int address, unsigned int datum)
{
unsigned int         adr_reg, adr_point, adr_gate;
   /* Discard the last four bits of the card
   address */
baseadr = baseadr & 0x0FF0;
   /* Address B0 to B4 of the counter */
address = address & 0x1F;

/* Load address pointer into address register */
   /* Adress of the address register */
adr_reg = baseadr | 0x0008;
   /* Value of the address register R0 to R2 =
   address of the counter without B0 and B1 */
adr_point = address >> 2;
   /* Load address register */
outpw (adr_reg, adr_point);

/* Calculate port address */
switch (axis)
       {
       case 1:
       adr_gate = baseadr | (address & 0x03);
       break;
       case 2:
       adr_gate = (baseadr | 0x0004) | (address & 0x03);
       break;
       }

   /* Write <datum> to the counter */
outpw (adr_gate, datum);
}

/*------------------------------------------------------
                       read_g26
 -------------------------------------------------------
 This function reads a value from a 16-bit
 register of a counter.
 -----------------------Parameters---------------------
 baseadr   basic address A0 to A9 of the IK 121
 axis      axis select - axis 1 or axis 2
 address   address of the counter B0 to B4
 adr_reg   address of the address register on the IK 121
 adr_point the part of the address indicated by the
           address register (R0 bis R2)
 adr_gate  port address in which <datum> is written
 ------------------------------------------------------*/
unsigned int read_g26 (unsigned int baseadr,
               unsigned char axis, unsigned int address)
{
unsigned int adr_reg, adr_point, adr_gate;
   /* Discard the last four bits of the card
   address */
baseadr = baseadr & 0x0FF0;
   /* Address B0 to B4 of the counter */
address = address & 0x1F;

/* Load address pointer into address register */
   /* Address of the address register */
adr_reg = baseadr | 0x0008;
   /* Value of the address register R0 to R2 =
   address of the counter without B0 and B1 */
adr_point = address >> 2;
   /* Load address register */
outpw (adr_reg, adr_point);

/* Calculate port address */
switch (axis)
       {
       case 1:
       adr_gate = baseadr | (address & 0x03);
       break;
       case 2:
       adr_gate = (baseadr | 0x0004) | (address & 0x03);
       break;
       }
   /* Read datum from the counter */
return(inpw(adr_gate));
}
/*------------------------------------------------------
                         soft_l0
 -------------------------------------------------------
 This function reads the measured value and stores
 it in data register 0.
 ------------------------------------------------------*/
void soft_l0 (unsigned int baseadr, unsigned char axis)
{
write_g26 (baseadr, axis, 0x0e, 0x01);
}


/*------------------------------------------------------
                         soft_l1
 -------------------------------------------------------
 This function reads the measured value and stores
 it in data register 1.
 -----------------------------------------------------*/
void soft_l1 (unsigned int baseadr, unsigned char axis)
{
write_g26 (baseadr, axis, 0x0e, 0x02);
}

/*------------------------------------------------------
                    read_count_value48
 -------------------------------------------------------
 This function reads 48 bits of a measured value.
 ------------------------------------------------------*/
double read_count_value48 (unsigned int baseadr,
                  unsigned char axis, unsigned char reg)
{
unsigned int field[3];
double count_value48;

switch (reg)
       {
       case 0:
       field[0] = read_g26 (baseadr, axis, 0);
       field[1] = read_g26 (baseadr, axis, 2);
       field[2] = read_g26 (baseadr, axis, 4);
       break;
       case 1:
       field[0] = read_g26 (baseadr, axis, 6);
       field[1] = read_g26 (baseadr, axis, 8);
       field[2] = read_g26 (baseadr, axis, 10);
       break;
       }

if (field[2] & 0x8000)
		 count_value48 = (double)((field[0]-65535L) +
		 65536.0*(field[1]-65535L)+
       4294967296.0*(field[2]-65535L)-1);
       else
       count_value48 = (double)(field[0] +
       65536.0*field[1] +
       4294967296.0*field[2]);


return (count_value48);
}
/*------------------------------------------------------
                    read_count_value32
 -------------------------------------------------------
 This function reads 32 bits of a measured value.
 ------------------------------------------------------*/
long read_count_value32 (unsigned int baseadr,
                  unsigned char axis, unsigned char reg)
{
union 	mapper
        {
        long field0;
        unsigned int field1[2];
        }buffer;

switch (reg)
       {
       case 0:
       buffer.field1[0] = read_g26 (baseadr, axis, 0);
       buffer.field1[1] = read_g26 (baseadr, axis, 2);
       break;
       case 1:
       buffer.field1[0] = read_g26 (baseadr, axis, 6);
       buffer.field1[1] = read_g26 (baseadr, axis, 8);
       break;
       }
return (buffer.field0);
}


/*------------------------------------------------------
                         latched
 -------------------------------------------------------
 This function checks whether a measured value is
 latched in data register 0 or 1.
 ------------------------------------------------------*/

unsigned char latched (unsigned int baseadr,
                  unsigned char axis, unsigned char reg)
{
unsigned char result;
switch (reg)
       {
       case 0:
       result = (unsigned char)
                (read_g26 (baseadr, axis, 14) & 0x01);
       break;
       case 1:
       result = (unsigned char)
                (read_g26 (baseadr, axis, 14) & 0x02);
       break;
       }
return (result);
}


/*------------------------------------------------------
                       poll_latch
 -------------------------------------------------------
 This function polls until a measured value is
 latched in data register 0 or 1.
 ------------------------------------------------------*/

void poll_latch (unsigned int baseadr,unsigned char axis,
                 unsigned char reg)
{
switch (reg)
       {
       case 0:
       while (latched (baseadr, axis, 0) == 0)
       ;
       break;

       case 1:
       while (latched (baseadr, axis, 1) == 0)
       ;
       break;
       }
}

