/*
   -----------------------------------------------------------------------
   -  DR JOHANNES HEIDENHAIN GmbH, Traunreut, West Germany               -
   -                                                                     -
   -  ik120.c           Id.No.:F90 BE 271 208 --                         -
   -                                                                     -
   -  For:                                                               -
   -  2 Counter and latch counter                                        -
   -  Interface cards                                                    -
   -  Id.Nr.: 490 BB 268 136 01                          Date: 15.03.95  - 
   -                                                                     -
   -----------------------------------------------------------------------
   -                                                                     -   
   -  This is a "c"-module intended for inclusion in user programs.      -
   -  ik120.c must be compiled to disk to produce the file ik120.obj     -
   -  before the user program can be compiled and linked.                -
   -                                                                     -
   -  e.g.: cl /AL /Oaxz /G2 /Fc Test.c ik120.obj                        -
   -                                                                     -
   ----------------------------------------------------------------------- */





   #include <ik120.h>      /* Globals, constants and variables are defined
                              in headerfile ik120.h                        */

 

   unsigned char reg_stat[2];



/* ===================
          RD        
   =================== */

   unsigned char RD (unsigned seg, unsigned off)

   {
   return ( (unsigned char) *(unsigned char *)POINTER(seg,off));
   }




/* ===================
           RDL       
   =================== */

   unsigned long RDL (unsigned seg, unsigned off)

   {
   return ( (unsigned long) *(unsigned long *)POINTER(seg,off));
   }




/* ===================
           WR        
   =================== */

   void WR (unsigned seg, unsigned off, unsigned char value)

   {
   *(char *)POINTER(seg,off) = (char)value;
   }
      



/* ####################################################################### */

   void Init_Interface(counter Counter)

/* Initialises the Counter on the interface card.
   Counter 0 and 1 must be initialised with seperate procedure calls:
        Init_Interface(C_0)
        Init_Interface(C_1)

   ---Initialise interface card---                                         */
   {
   WR (boardseg, boardoff+0x20, 0x8f);        /* Control Register 1        */
   WR (boardseg, boardoff+0x20, 0x83);
   WR (boardseg, boardoff+0x20, 0x8f);
   reg_stat[0] = 0x8f;                        /* save Status               */
   WR (boardseg, boardoff+0x30, 0x80);        /* Control Register 2        */
   WR (boardseg, boardoff+0x30, 0x20);
   reg_stat[1] = 0x20;                        /* save Status               */
   WR (boardseg, boardoff+0x40, 0xff);        /* Intervall-Value LOW       */
   WR (boardseg, boardoff+0x50, 0xff);        /* Intervall-Value HIGH      */


/* ----Initialise counter   -----                                          */

   WR (boardseg, boardoff+Counter+0x0f, 0x00);  /* Initialising Register   */
   WR (boardseg, boardoff+Counter+0x0d, 0xff);  /* Delete Status           */
   WR (boardseg, boardoff+Counter+0x0c, 0x03);  /* Default counting method */
   WR (boardseg, boardoff+Counter+0x0b, 0x06);  /* Delete and stop counter */
   WR (boardseg, boardoff+Counter+0x0a, 0x00);  /* Ref-Impulse inactive    */
   RD (boardseg, boardoff+Counter+0x03);        /* Sp-Reg0 enable          */
   RD (boardseg, boardoff+Counter+0x07);        /* Sp-Reg1 enable          */
   }                            


/* ####################################################################### */

   void Reset_Status(counter Counter)

/* Resets the status byte of the selected Counter                          */

   {
   WR (boardseg, boardoff+Counter+0x0d, 0xff);
   }



/* ####################################################################### */

   int Read_Status(counter Counter)

/*  Reads the status byte. Evalation of the result must be carried out
    seperately.                                                            */

   {
   return (RD (boardseg, boardoff+Counter+0x0e));
   }



/* ####################################################################### */

   int Signal_Error(counter Counter)

/* Reads the status byte and returns true when a signal amplitude
   or frequency error has occured.                                         */

   {
   int zeichen = 0;
   zeichen = RD (boardseg, boardoff+Counter+0x0e);
      if (zeichen & 0x08)
         return (1);
      else
         return (0);
   }



/* ####################################################################### */

   int Read_ScanReg(counter Counter)

/* Reads value from scan-register.                                         */

   {
   return RD (boardseg, boardoff+Counter+0x0f);
   }



/* ####################################################################### */

   int Latched(counter Counter, latch Latch)

/* Returns true if the latch bit has been set.                             */

   {
   switch (Latch)
      {
      case L_0:
         return (Read_Status (Counter) & 0x01);
         break;
      case L_1:
         return (Read_Status (Counter) & 0x02);
         break;
      }
   }



/* ####################################################################### */

   void Reset_uas(counter Counter)
                                                                           
/* Resets the error bit for frequency or amplitude errors on selected Counter */


   {
   switch (Counter)
      {
      case C_0:
      	reg_stat[0] |= 0x04;
      	WR (boardseg, boardoff+0x20, reg_stat[0]);
      	reg_stat[0] &= 0xfb;
      	WR (boardseg, boardoff+0x20, reg_stat[0]);
      	reg_stat[0] |= 0x04;
      	WR (boardseg, boardoff+0x20, reg_stat[0]);   /* Control_Register 1*/
      	break;
      case C_1:
      	reg_stat[0] |= 0x08;
      	WR (boardseg, boardoff+0x20, reg_stat[0]);
      	reg_stat[0] &= 0xf7;
      	WR (boardseg, boardoff+0x20, reg_stat[0]);
      	reg_stat[0] |= 0x08;
      	WR (boardseg, boardoff+0x20, reg_stat[0]);   /* Control_Register 1*/
         break;
      }
   }



/* ####################################################################### */

   void Interpolation(counter Counter, interpol Interpol)

/* Sets the specified interpolation.                                       
   e.g. Interpolation(C_0,I_50) sets Counter 0 to 50 fold interpolation so
        that 200 counts per grating period occur.                                    */


   {
   switch(Counter)
      {
      case C_0:
         switch(Interpol)
            {
            case I_50:
               reg_stat[0] |= 0x01;
               break;
            case I_25:
               reg_stat[0] &= 0xfe;
               break;
            }
         break;
      case C_1:
         switch(Interpol)
            {
            case I_50:
               reg_stat[0] |= 0x02;
               break;
            case I_25:
               reg_stat[0] &= 0xfd;
               break;
            }
         break;
      }
   WR (boardseg, boardoff+0x20, reg_stat[0]);      /*  Control_Register 1  */
   }



/* ####################################################################### */

   void Init_Latch(control Control, eval Eval, int interval)

/* Initialises the latch counter.
   e.g. Init_Latch(RI_start,fourfold,interval);
                  Means: Start Counter with RI
                         Fourfold evaluation
                         Latch value defined by interval                   */



   {
   unsigned char value_lo, value_hi;


      switch(Control)
         {
         case C_start:
            reg_stat[1] &= 0x0f;
            reg_stat[1] |= 0x10;
            break;
         case C_stop:
            reg_stat[1] &= 0x0f;
            reg_stat[1] |= 0x20;
            break;
         case C_reset:
            reg_stat[1] &= 0x0f;
            reg_stat[1] |= 0x80;
            break;
         case C_latch:
            reg_stat[1] &= 0x0f;
            reg_stat[1] |= 0x70;
            break;
         case C_load:
            reg_stat[1] &= 0x0f;
            reg_stat[1] |= 0xb0;
            break;
         case RI_start:
            reg_stat[1] &= 0x0f;
            reg_stat[1] |= 0x50;
            break;
         case RI_stop:
            reg_stat[1] &= 0x0f;
            reg_stat[1] |= 0x60;
            break;
         case RI_reset:
            reg_stat[1] &= 0x0f;
            reg_stat[1] |= 0xa0;
            break;
         case RI_latch:
            reg_stat[1] &= 0x0f;
            reg_stat[1] |= 0xf0;
            break;
         case RI_load:
            reg_stat[1] &= 0x0f;
            reg_stat[1] |= 0xd0;
            break;
         case Strobe_latch:
            reg_stat[1] &= 0x0f;
            reg_stat[1] |= 0xe0;
            break;
         }


      switch(Eval)
         {
         case onefold:
            reg_stat[0] &= 0xcf;
            reg_stat[0] |= 0x20;
            break;
         case twofold:
            reg_stat[0] &= 0xcf;
            reg_stat[0] |= 0x10;
            break;
         case fourfold:
            reg_stat[0] &= 0xcf;
            break;
         }


   interval -= 1;
   if (interval < 1)
      interval = 1;
   value_lo = (unsigned char)interval;
   value_hi = (unsigned char)(interval >> 8);

   WR (boardseg, boardoff+0x20, reg_stat[0]);      /*  Control_Register 1  */
   WR (boardseg, boardoff+0x30, reg_stat[1]);      /*  Control_Register 2  */
   WR (boardseg, boardoff+0x40, value_lo);         /*  Interval value lsb  */
   WR (boardseg, boardoff+0x50, value_hi);         /*  Interval value msb  */
   }



/* ####################################################################### */

   void Latch_Enable(lcontrol Lcontrol)


   {
      switch(Lcontrol)
         {
         case Internal:
            reg_stat[1] |= 0x01;
            break;
         case ExternalX1:
            reg_stat[1] |= 0x02;
            break;
         case ExternalX2:
            reg_stat[1] |= 0x04;
            break;
         case ExternalX1X2:
            reg_stat[1] |= 0x06;
            break;
         case LatchOut:
            reg_stat[1] |= 0x08;
            break;
         }

   WR (boardseg, boardoff+0x30, reg_stat[1]);      /*  Control_Register 2  */
   }


/* ####################################################################### */

   void Latch_Disable(lcontrol Lcontrol)

   {
      switch(Lcontrol)
         {
         case Internal:
            reg_stat[1] &= 0xfe;
            break;
         case ExternalX1:
            reg_stat[1] &= 0xfd;
            break;
         case ExternalX2:
            reg_stat[1] &= 0xfb;
            break;
         case ExternalX1X2:
            reg_stat[1] &= 0xf9;
            break;
         case LatchOut:
            reg_stat[1] &= 0xf7;
            break;
         }

   WR (boardseg, boardoff+0x30, reg_stat[1]);      /*  Control_Register 2  */
   }



/* ####################################################################### */

   void Write_Strobe()

/* ------------------------------------------------------------------------
   Writes a Strobe signal to Interval Counter of the selected card.   
   Can be used to trigger simultaneous software latching of both axes 
   by programing interval counter with Strobe_latch.                  
   ------------------------------------------------------------------------*/

   {
   WR (boardseg, boardoff+0x60, 0xff);
   }


/* ####################################################################### */

   void Init_Counter(counter Counter, control Control, eval Eval,
                     direction Direction, method Method)

/* ------------------------------------------------------------------------
   Initialises the specified counter.                                     
   e.g. Init_Counter(C_0,C_start,fourfold,normal,linear);                     
        Result:  Starts counter C_0
                 Each edge will be counted e.g. four counts per TTL signal
                 Normal count direction					  
                 Linear counting method					 
   ------------------------------------------------------------------------*/

   {
   unsigned char befehl,reg_a,reg_b;

   WR (boardseg, boardoff+Counter+0x0f, 0x00);  /* Initialising_Register   */
   WR (boardseg, boardoff+Counter+0x0d, 0xff);  /*   Delete status_bits    */



      switch(Eval)
         {
         case onefold:
            switch(Direction)
               {
               case normal:
                  switch(Method)
                     {
                     case linear:
                        befehl = 0x00;
                        break;
                     case arc:
                        befehl = 0x04;
                        break;
                     }
               break;
               case inverse:
                  switch(Method)
                     {
                     case linear:
                        befehl = 0x08;
                        break;
                     case arc:
                        befehl = 0x0c;
                        break;
                     }
                  break;
               }
            break;
         case twofold:
            switch(Direction)
               {
               case normal:
                  switch(Method)
                     {
                     case linear:
                        befehl = 0x01;
                        break;
                     case arc:
                        befehl = 0x05;
                        break;
                     }
                  break;
               case inverse:
                  switch(Method)
                     {
                     case linear:
                        befehl = 0x09;
                        break;
                     case arc:
                        befehl = 0x0d;
                        break;
                     }
                  break;
               }
            break;
         case fourfold:
            switch(Direction)
               {
               case normal:
                  switch(Method)
                     {
                     case linear:
                        befehl = 0x03;
                        break;
                     case arc:
                        befehl = 0x07;
                        break;
                     }
                  break;
               case inverse:
                  switch(Method)
                     {
                     case linear:
                        befehl = 0x0b;
                        break;
                     case arc:
                        befehl = 0x0f;
                        break;
                     }
                  break;
               }
            break;
         }

   WR (boardseg, boardoff+Counter+0x0c, befehl);  /*Operating mode_register*/

   reg_a = 0x00;
   reg_b = 0x00;

      switch (Control)
         {
         case RI_start:
            reg_a = 0x01;
            break;
         case RI_stop:
            reg_a = 0x02;
            break;
         case RI_reset:
            reg_a = 0x04;
            break;
         case RI_reset_start:
            reg_a = 0x05;
            break;
         case RI_reset_stop:
            reg_a = 0x06;
            break;
         case RI_latch:
            reg_a = 0x08;
            break;
         case RI_latch_start:
            reg_a = 0x09;
            break;
         case RI_latch_stop:
            reg_a = 0x0a;
            break;
         case RI_latch_reset:
            reg_a = 0x0c;
            break;
         case RI_latch_reset_start:
            reg_a = 0x0d;
            break;
         case RI_latch_reset_stop:
            reg_a = 0x0e;
            break;
         case C_start:
            reg_b = 0x01;
            break;
         case C_stop:
            reg_b = 0x02;
            break;
         case C_reset:
            reg_b = 0x04;
            break;
         case C_reset_start:
            reg_b = 0x05;
            break;
         case C_reset_stop:
            reg_b = 0x06;
            break;
         }

   WR (boardseg, boardoff+Counter+0x0a, reg_a);     /*  Ref.mark_register  */
   WR (boardseg, boardoff+Counter+0x0b, reg_b);     /*  Command_register   */
   }



/* ####################################################################### */

   void Latch_Count(counter Counter, latch Latch)

/* ------------------------------------------------------------------------
   Latches the specified Counter
   The count is stored to the specified register
   ------------------------------------------------------------------------*/

   {
   WR (boardseg, boardoff+Counter+Latch+0x08, 0x00);
   }



/* ####################################################################### */

   unsigned long Read_Count(counter Counter, latch Latch)

/* ------------------------------------------------------------------------
   Reads the current value from the specified register			
   Latching must be carried out separately 				
   ------------------------------------------------------------------------*/

   {
   return (RDL (boardseg, boardoff+Counter+Latch*4));
   }



/* ####################################################################### */

   unsigned long Soft_Count(counter Counter, latch Latch)

/* ------------------------------------------------------------------------
   Latches and reads the current value from the specified register	 
   ------------------------------------------------------------------------*/

   {
   WR (boardseg, boardoff+Counter+Latch+0x08, 0x00);
   return (RDL (boardseg, boardoff+Counter+Latch*4));
   }



/* ####################################################################### */

   void Clear_Int()

/* ------------------------------------------------------------------------
   Resets the Interrupt register.					 
   Must be included in the interrupt service routine.			 
   ------------------------------------------------------------------------*/

   {
   WR (boardseg, boardoff+0x70, 0x00);
   }



/* ####################################################################### */

   void Interrupt_Enable(interr Interr)

/* ------------------------------------------------------------------------
   Enable interrupt operating mode. 
   e.g. Interrupt_Enable(Int_0) means that the latch counter  	  
        will generate an interrupt (The interrupt jumper must be set).    
        Interrupt_Enable(Int_1) means that if the external latch 
        input is enabled an interrupt will occur when a latch signal is   
        detected                                                          
   ------------------------------------------------------------------------*/


   {
   switch (Interr)
      {
      case Int_0:
         reg_stat[0] |= 0x40;
         break;
      case Int_1:
         reg_stat[0] |= 0x80;
         break;
      }


   WR (boardseg, boardoff+0x20, reg_stat[0]);
   }

