UNIT IK121_1;

{-------------------------------------------------------}
{ DR. JOHANNES HEIDENHAIN GmbH, Traunreut, Germany      }
{                                                       }
{ Driver Unit for IK121                                 }
{                                                       }
{ V 1.01                                                }
{ April 1995                                            }
{ V 1.02                                                }
{ March 1996                                            }
{ V 1.04                                                }
{ Mai 1998                                              }
{-------------------------------------------------------}

{$N+,E+}
{$V+}
{$R+}

interface

uses ik121_0,crt;

CONST
      IRQ5  =$0D;
      IRQ9  =$71;
      IRQ10 =$72;
      IRQ11 =$73;
      IRQ12 =$74;
      IRQ15 =$77;

TYPE
   softcommand   = record
                     start  : boolean;
                     stop  : boolean;
                     clear  : boolean;
                     latch0 : boolean;
                     latch1 : boolean;
                     latch2 : boolean;
                     clrfreq: boolean;
                     clrstat: boolean;
                   end;
   refcommand    = record
                     ristart : boolean;
                     ristop : boolean;
                     riclear : boolean;
                     riclear2: boolean;
                     rilatch : boolean;
                     rilatch2: boolean;
                   end;
   initlatch     = record
                     en_l0_reg0     : boolean;
                     en_latch2_reg0 : boolean;
                     en_timer_reg0  : boolean;
                     en_l1_reg1     : boolean;
                     en_latch2_reg1 : boolean;
                     en_timer_reg1  : boolean;
                   end;
  initsync       = record
                     en_l0_axis2     : boolean;
                     en_latch2_axis2 : boolean;
                     en_timer_axis2  : boolean;
                   end;
  initmain       = record
                    mode1024   : boolean;
                    en_timer   : boolean;
                    en_48bit   : boolean;
                    onefold    : boolean;
                    twofold    : boolean;
                    fourfold   : boolean;
                    arcmode    : boolean;
                    arc180000  : boolean;
                    sel_ri_1st : boolean;
                    sel_ri_2nd : boolean;
                  end;
  initintrpt    = record
                    reg0    : boolean;
                    reg1    : boolean;
                    timer   : boolean;
                    port    : boolean;
                  end;

  intstate    = record
                  l0        : boolean;
                  l1        : boolean;
                  timer     : boolean;
                  pendl0    : boolean;
                  pendl1    : boolean;
                  pendtimer : boolean;
                end;
  countstate  = record
                  latch0    : boolean;
                  latch1    : boolean;
                  stop     : boolean;
                  sda       : boolean;
                  error_freq: boolean;
                  ref_activ : boolean;
                end;
  signalstate = record
                  ad00       : word;
                  ad90       : word;
                  amp_act    : byte;
                  amp_min    : byte;
                end;

   g26_pointr    = ^g26_record;
   g26_record    = record
                     baseadr       : word;
                     axis          : byte;
                     register0     : comp;    (* Increments *)
                     register1     : comp;
                     direction     : boolean;

                     softcomm      : softcommand;
                     refcomm       : refcommand;

                     inimain       : initmain;
                     inilatch      : initlatch;
                     inisync       : initsync;
                     iniint        : initintrpt;

                     cstatus       : countstate;
                     sstatus       : signalstate;
                     istatus       : intstate;

                     timer         : word;
                     offset00      : integer;
                     offset90      : integer;
                 end;

  ik121_pointr = ^ik121_record;
  ik121_record = record
                   axis    : array[1..2] of g26_pointr;
                 end;

  storage = record
               axis  : array[1..2] of g26_record;
             end;

PROCEDURE Write_offset(baseadr:word;axis:byte;offx,offy:integer);

FUNCTION  Look_for_IK121(board:IK121_pointr):boolean; (* Test for hardware *)
PROCEDURE Init_ik121(board:IK121_pointr);
FUNCTION  Read_Adr:word;                        (* Reads  IK121.INI *)
PROCEDURE Write_Adr(address:word);              (* Writes IK121.INI *)
PROCEDURE Read_Signal_Status(pointr:G26_pointr);
PROCEDURE Read_Count_Status(pointr:G26_pointr);
PROCEDURE Read_Int_Status(pointr:G26_pointr);
PROCEDURE Init_Handler(pointr:g26_pointr);
PROCEDURE Comm_Handler(pointr:g26_pointr);

(* Direct hardware access procedures *)
PROCEDURE Soft_Latch0(pointr:g26_pointr);
PROCEDURE Soft_Latch1(pointr:g26_pointr);
PROCEDURE Read_Reg0(pointr:G26_pointr);
PROCEDURE Read_Reg1(pointr:G26_pointr);
PROCEDURE Poll_Reg0(pointr:G26_pointr);
PROCEDURE Poll_Reg1(pointr:G26_pointr);

(* Interrupt procedures *)
PROCEDURE En_Int(Intrpt:byte);
PROCEDURE Dis_Int(Intrpt:byte);
PROCEDURE Clear_Int;

(* Read/write offset from EEPROM *)
PROCEDURE Load_offset(board:IK121_pointr;var error:boolean);
PROCEDURE Store_offset(board:IK121_pointr;var error:boolean);

(* EEPROM pot procedures *)
PROCEDURE Poti_default(pointr:ik121_pointr;var error:boolean);
FUNCTION  Read_phasepoti(pointr:ik121_pointr;axis:byte;var error:boolean):byte;
FUNCTION  Read_sympoti(pointr:ik121_pointr;axis:byte;var error:boolean):byte;
PROCEDURE Write_phasepoti(pointr:ik121_pointr;axis,value:byte;var error:boolean);
PROCEDURE Write_sympoti(pointr:ik121_pointr;axis,value:byte;var error:boolean);
PROCEDURE Write_offset00(pointr:ik121_pointr;axis,value:integer);
PROCEDURE Write_offset90(pointr:ik121_pointr;axis,value:integer);
PROCEDURE Turn_phasepoti(pointr:ik121_pointr;axis,turns:byte;updown:boolean;var error:boolean);
PROCEDURE Turn_sympoti(pointr:ik121_pointr;axis,turns:byte;updown:boolean;var error:boolean);
PROCEDURE Turn_offsetdg00(pointr:ik121_pointr;axis,turns:byte;updown:boolean);
PROCEDURE Turn_offsetdg90(pointr:ik121_pointr;axis,turns:byte;updown:boolean);
PROCEDURE Store_potis(pointr:ik121_pointr;var error:boolean);

(* EEPROM ROM procedures *)
PROCEDURE Rom_write(pointr:ik121_pointr;adr:word;data :byte;var error :boolean);
FUNCTION  Rom_read(pointr:ik121_pointr;adr:word;var error:boolean):byte;

Implementation
{----------------------------------------------------------}
PROCEDURE Write_offset(baseadr:word;axis:byte;offx,offy:integer);
  VAR
    timeout,buffer,
    off0,off9 : word;
  BEGIN
     timeout:=0;
     repeat
       timeout:=timeout+1;
       buffer:=read_g26(baseadr,axis,28);
     until ((buffer and $0060)=$0000) or (timeout=1000);  (* V1.02 *)
    if offx<0 then
      begin
        if offx<-63 then offx:=-63;
        offx:=127+offx;
        off0:=offx or $0040;  (* Minus *)
      end
    else
      begin
        if offx>63 then offx:=63;
        off0:=offx;
      end;
    if offy<0 then
      begin
        if offy<-63 then offy:=-63;
        offy:=127+offy;
        off9:=offy or $0040;
      end
    else
      begin
        if offy>63 then offy:=63;
        off9:=offy;
      end;
    Write_g26(baseadr,axis,22,off0);
    Write_g26(baseadr,axis,24,off9);
  END;
{----------------------------------------------------------}
PROCEDURE Soft_Latch0(pointr:g26_pointr);
  BEGIN
    write_g26(pointr^.baseadr,pointr^.axis,14,$0001);
  END;
{----------------------------------------------------------}
PROCEDURE Soft_Latch1(pointr:g26_pointr);
  BEGIN
    write_g26(pointr^.baseadr,pointr^.axis,14,$0002);
  END;
{----------------------------------------------------------}
PROCEDURE Read_Reg0(pointr:G26_pointr);
  VAR
    value : comp;
  BEGIN
    if Latched(pointr^.baseadr,pointr^.axis,0) then
      begin
        if pointr^.inimain.en_48bit then
          begin
            value:=Read_Count_value48(pointr^.baseadr,pointr^.axis,0,pointr^.direction);
          end
        else
          begin
            value:=Read_Count_value32(pointr^.baseadr,pointr^.axis,0,pointr^.direction);
          end;

        if pointr^.inimain.mode1024 then
          begin
            pointr^.register0:=value;
          end
        else
          begin
            pointr^.register0:=value/1024
          end;

      end;
  END;
{----------------------------------------------------------}
PROCEDURE Read_Reg1(pointr:G26_pointr);
  VAR
    value : comp;
  BEGIN
    if Latched(pointr^.baseadr,pointr^.axis,1) then
      begin
        if pointr^.inimain.en_48bit then
          begin
            value:=Read_Count_value48(pointr^.baseadr,pointr^.axis,1,pointr^.direction);
          end
        else
          begin
            value:=Read_Count_value32(pointr^.baseadr,pointr^.axis,1,pointr^.direction);
          end;

        if pointr^.inimain.mode1024 then
          begin
            pointr^.register1:=value;
          end
        else
          begin
            pointr^.register1:=value/1024;
          end;

      end;
  END;
{----------------------------------------------------------}
PROCEDURE Poll_Reg0(pointr:G26_pointr);
  VAR
    eing : char;
    ende : boolean;
  BEGIN
    ende:=false;
    repeat
      if keypressed then
        begin
          eing:=readkey;  (* Stop with Ctrl C *)
          if eing=#03 then
            begin
              ende:=true;
            end;
        end;
    until (Latched(pointr^.baseadr,pointr^.axis,0) or ende);
    Read_Reg0(pointr);
  END;
{----------------------------------------------------------}
PROCEDURE Poll_Reg1(pointr:G26_pointr);
  VAR
    eing : char;
    ende : boolean;
  BEGIN
    ende:=false;
    repeat
      if keypressed then
        begin
          eing:=readkey;  (* Stop with Ctrl C *)
          if eing=#03 then
            begin
              ende:=true;
            end;
        end;
    until (Latched(pointr^.baseadr,pointr^.axis,1) or ende);
    Read_Reg1(pointr);
  END;
{----------------------------------------------------------}
FUNCTION G26_Soft(pointr:G26_pointr):word;
  VAR
    control_buffer : word;
  BEGIN
      control_buffer:=$0000;
      if (pointr^.softcomm.start) then
        begin
          control_buffer:=control_buffer or $0008;
          pointr^.softcomm.start:=false;
        end;
      if (pointr^.softcomm.stop) then
        begin
          control_buffer:=control_buffer or $0010;
          pointr^.softcomm.stop:=false;
        end;
      if (pointr^.softcomm.clear) then
        begin
          control_buffer:=control_buffer or $0020;
          pointr^.softcomm.clear:=false;
        end;
      if (pointr^.softcomm.latch0) then
        begin
          control_buffer:=control_buffer or $0001;
          pointr^.softcomm.latch0:=false;
        end;
      if (pointr^.softcomm.latch1) then
        begin
          control_buffer:=control_buffer or $0002;
          pointr^.softcomm.latch1:=false;
        end;
      if (pointr^.softcomm.latch2) then
        begin
          control_buffer:=control_buffer or $0004;
          pointr^.softcomm.latch2:=false;
        end;
      if (pointr^.softcomm.clrfreq) then
        begin
          control_buffer:=control_buffer or $0040;
          pointr^.softcomm.clrfreq:=false;
        end;
      if (pointr^.softcomm.clrstat) then
        begin
          control_buffer:=control_buffer or $0080;
          pointr^.softcomm.clrstat:=false;
        end;
      G26_Soft:=control_buffer;
   END;
{----------------------------------------------------------}
FUNCTION G26_Ref(pointr:G26_pointr):word;
  VAR
    ref_buffer : word;
  BEGIN
    ref_buffer:=$0000;
      if (pointr^.refcomm.ristart) then
        begin
          ref_buffer:=ref_buffer or $0001;
          pointr^.refcomm.ristart:=false;
        end;
      if (pointr^.refcomm.ristop) then
        begin
         ref_buffer:=ref_buffer or $0002;
         pointr^.refcomm.ristop:=false;
       end;
      if (pointr^.refcomm.riclear) then
        begin
          ref_buffer:=ref_buffer or $0004;
          pointr^.refcomm.riclear:=false;
        end;
      if (pointr^.refcomm.riclear2) then
        begin
          ref_buffer:=ref_buffer or $0020;
          pointr^.refcomm.riclear2:=false;
        end;
      if (pointr^.refcomm.rilatch) then
        begin
          ref_buffer:=ref_buffer or $0008;
          pointr^.refcomm.rilatch:=false;
        end;
      if (pointr^.refcomm.rilatch2) then
        begin
          ref_buffer:=ref_buffer or $0010;
          pointr^.refcomm.rilatch2:=false;
        end;
      G26_ref:=ref_buffer;
   END;
{----------------------------------------------------------}
FUNCTION G26_Latch(pointr:G26_pointr):word;
  VAR
    latch_buffer : word;
  BEGIN
    latch_buffer:=$0000;
      if (pointr^.inilatch.en_l0_reg0) then
        begin
          if pointr^.axis=1 then
            begin
              (* Master *)
              latch_buffer:=latch_buffer or $0002;
            end
          else
            begin
              (* Slave *)
              latch_buffer:=latch_buffer or $0001;
            end;
        end;
      if (pointr^.inilatch.en_latch2_reg0) then latch_buffer:=latch_buffer or $0004;
      if (pointr^.inilatch.en_timer_reg0) then latch_buffer:=latch_buffer or $0008;
      (* L1 slave only *)
      if (pointr^.inilatch.en_l1_reg1) then latch_buffer:=latch_buffer or $0010;
      if (pointr^.inilatch.en_latch2_reg1) then latch_buffer:=latch_buffer or $0040;
      if (pointr^.inilatch.en_timer_reg1) then latch_buffer:=latch_buffer or $0080;
      G26_Latch:=latch_buffer;
   END;
{----------------------------------------------------------}
FUNCTION G26_Sync(pointr:G26_pointr):word;
  VAR
    syn_buffer   : word;
  BEGIN
    syn_buffer:=$0000;
    if pointr^.axis=1 then
      begin
        if (pointr^.inisync.en_l0_axis2) then syn_buffer:=syn_buffer or $0100;
        if (pointr^.inisync.en_latch2_axis2) then syn_buffer:=syn_buffer or $0200;
        if (pointr^.inisync.en_timer_axis2) then syn_buffer:=syn_buffer or $0400;
      end
    else
      begin
        (* scl,sda=1 *)
        syn_buffer:=syn_buffer or $0000;  (* V1.02 *)
      end;
    G26_Sync:=syn_buffer;
   END;
{----------------------------------------------------------}
FUNCTION G26_Main(pointr:G26_pointr):word;
  VAR
    betr_buffer   : word;
  BEGIN
    betr_buffer:=$0000;
    if (pointr^.inimain.mode1024) then betr_buffer:=betr_buffer or $0001;
    if (pointr^.inimain.en_timer) then betr_buffer:=betr_buffer or $0004;
    if (pointr^.inimain.en_48bit) then betr_buffer:=betr_buffer or $0040;
    if (pointr^.inimain.fourfold) then
      begin
        pointr^.inimain.onefold:=false;
        pointr^.inimain.twofold:=false;
        betr_buffer:=betr_buffer or $0300;
      end;
    if (pointr^.inimain.twofold) then
      begin
        pointr^.inimain.onefold:=false;
        pointr^.inimain.fourfold:=false;
        betr_buffer:=betr_buffer or $0200;
      end;
    if (pointr^.inimain.onefold) then
      begin
        pointr^.inimain.twofold:=false;
        pointr^.inimain.fourfold:=false;
      end;
    if (pointr^.inimain.arcmode) then betr_buffer:=betr_buffer or $0400;
    if (pointr^.inimain.arc180000) then betr_buffer:=betr_buffer or $0800;
    if (pointr^.inimain.sel_ri_1st) then betr_buffer:=betr_buffer or $4000;
    if (pointr^.inimain.sel_ri_2nd) then betr_buffer:=betr_buffer or $8000;
    G26_Main:=betr_buffer;
  END;
{----------------------------------------------------------}
FUNCTION G26_Int(pointr:G26_pointr):word;
  VAR
    int_buffer   : word;
  BEGIN
    int_buffer:=$0000;
    if (pointr^.iniint.reg0) then int_buffer:=int_buffer or $0001;
    if (pointr^.iniint.reg1) then int_buffer:=int_buffer or $0002;
    if (pointr^.iniint.timer) then int_buffer:=int_buffer or $0004;
    if (pointr^.iniint.port) then int_buffer:=int_buffer or $0008;
    G26_Int:=int_buffer;
END;
{----------------------------------------------------------}
PROCEDURE Init_Handler(pointr:g26_pointr);
  VAR
    buffer,timeout,
    lat_buffer,syn_buffer,
    betr_buffer,int_buffer,
    timer_buffer : word;
  BEGIN
     (* Fixed Values *)
     write_g26(pointr^.baseadr,pointr^.axis,30,$0000);
     write_g26(pointr^.baseadr,pointr^.axis,28,$0008);

     (* Main *)
     betr_buffer:=G26_Main(pointr);
     write_g26(pointr^.baseadr,pointr^.axis,12,betr_buffer);

     (* Latch and sync *)
     lat_buffer:=G26_latch(pointr);
     syn_buffer:=G26_sync(pointr);
     write_g26(pointr^.baseadr,pointr^.axis,18,lat_buffer+syn_buffer);

     (* Interrupt *)
     int_buffer:=G26_int(pointr);
     write_g26(pointr^.baseadr,pointr^.axis,20,int_buffer);

     (* Timer Value *)
     timer_buffer:=pointr^.timer;
     write_g26(pointr^.baseadr,pointr^.axis,26,timer_buffer);

     (* Offset DG00,DG90 *)
     write_offset(pointr^.baseadr,pointr^.axis,
                  pointr^.offset00,pointr^.offset90);
   END;
{----------------------------------------------------------}
PROCEDURE Comm_Handler(pointr:g26_pointr);
  VAR
    soft_buffer,ref_buffer,amp_buffer : word;
  BEGIN
    soft_buffer:=G26_soft(pointr);
    if (soft_buffer<>0) then
      begin
        write_g26(pointr^.baseadr,pointr^.axis,14,soft_buffer);
      end;
    ref_buffer:=G26_ref(pointr);
    if (ref_buffer<>0) then
      begin
        write_g26(pointr^.baseadr,pointr^.axis,16,ref_buffer);
      end;
   END;
{----------------------------------------------------------}
PROCEDURE Read_Count_Status(pointr:G26_pointr);
  VAR
    buffer : word;
  BEGIN
    buffer:=Read_G26(pointr^.baseadr,pointr^.axis,14);
    if (buffer and $0001) = $0001 then pointr^.cstatus.Latch0:=true
                                  else pointr^.cstatus.Latch0:=false;
    if (buffer and $0002) = $0002 then pointr^.cstatus.Latch1:=true
                                  else pointr^.cstatus.Latch1:=false;
    if (buffer and $0010) = $0010 then pointr^.cstatus.stop:=true
                                  else pointr^.cstatus.stop:=false;
    if (buffer and $0020) = $0020 then pointr^.cstatus.sda:=true  (* V1.02 *)
                                  else pointr^.cstatus.sda:=false;
    if (buffer and $0040) = $0040 then pointr^.cstatus.error_freq:=true
                                  else pointr^.cstatus.error_freq:=false;
    if (buffer and $0100) = $0100 then pointr^.cstatus.ref_activ:=true
                                  else pointr^.cstatus.ref_activ:=false;
  END;
{----------------------------------------------------------}
PROCEDURE Read_Signal_Status(pointr:G26_pointr);
  VAR
    buffer   : word;
    timeout  : word;
  BEGIN
    buffer:=$0018;
    write_g26(pointr^.baseadr,pointr^.axis,28,buffer);
    buffer:=0;
    timeout:=0;
    repeat
      timeout:=timeout+1;
      buffer:=Read_G26(pointr^.baseadr,pointr^.axis,28);
      buffer:=buffer and $0010;
    until (buffer=$0010) or (timeout=$0FFF);
    pointr^.sstatus.ad00:=Read_G26(pointr^.baseadr,pointr^.axis,22);
    pointr^.sstatus.ad90:=Read_G26(pointr^.baseadr,pointr^.axis,24);
    buffer:=Read_G26(pointr^.baseadr,pointr^.axis,16);
    pointr^.sstatus.amp_act:=(buffer and $0300) shr 8;
    pointr^.sstatus.amp_min:=(buffer and $0C00) shr 10;
    buffer:=$0008;
    write_g26(pointr^.baseadr,pointr^.axis,28,buffer);
  END;
{----------------------------------------------------------}
PROCEDURE Read_Int_Status(pointr:G26_pointr);
  VAR
    buffer   : word;
    timeout  : word;
  BEGIN
    buffer:=Read_G26(pointr^.baseadr,pointr^.axis,20);

    if (buffer and $0100=$0100) then
      begin
        pointr^.istatus.l0:=true;
      end
    else
      begin
        pointr^.istatus.l0:=false;
      end;
   if (buffer and $0200=$0200) then
      begin
        pointr^.istatus.l1:=true;
      end
    else
      begin
        pointr^.istatus.l1:=false;
      end;
   if (buffer and $0400=$0400) then
      begin
        pointr^.istatus.timer:=true;
      end
    else
      begin
        pointr^.istatus.timer:=false;
      end;

    if (buffer and $0001=$0001) then
      begin
        pointr^.istatus.pendl0:=true;
      end
    else
      begin
        pointr^.istatus.pendl0:=false;
      end;
   if (buffer and $0002=$0002) then
      begin
        pointr^.istatus.pendl1:=true;
      end
    else
      begin
        pointr^.istatus.pendl1:=false;
      end;
   if (buffer and $0004=$0004) then
      begin
        pointr^.istatus.pendtimer:=true;
      end
    else
      begin
        pointr^.istatus.pendtimer:=false;
      end;
  END;
{----------------------------------------------------------}
PROCEDURE Clear_Int;
BEGIN
  port[$A0]:=$20;
  port[$20]:=$20;
END;
{----------------------------------------------------------}
PROCEDURE En_Int(Intrpt:byte);
BEGIN
  case Intrpt of
    IRQ5  : begin
              port[$21]:=port[$21] and $DF;
            end;
    IRQ9  : begin
              port[$A1]:=port[$A1] and $FD;
              port[$21]:=port[$21] and $FB;
            end;
    IRQ10 : begin
              port[$A1]:=port[$A1] and $FB;
              port[$21]:=port[$21] and $FB;
            end;
    IRQ11 : begin
              port[$A1]:=port[$A1] and $F7;
              port[$21]:=port[$21] and $FB;
            end;
    IRQ12 : begin
              port[$A1]:=port[$A1] and $EF;
              port[$21]:=port[$21] and $FB;
            end;
    IRQ15 : begin
              port[$A1]:=port[$A1] and $7F;
              port[$21]:=port[$21] and $FB;
            end;
  end;
END;
{----------------------------------------------------------}
PROCEDURE Dis_Int(Intrpt:byte);
BEGIN
  case Intrpt of
    IRQ5  : begin
              port[$21]:=port[$21] or $20;
            end;
    IRQ9  : begin
              port[$A1]:=port[$A1] or $02;
            end;
    IRQ10 : begin
              port[$A1]:=port[$A1] or $04;
            end;
    IRQ11 : begin
              port[$A1]:=port[$A1] or $08;
            end;
    IRQ12 : begin
              port[$A1]:=port[$A1] or $10;
            end;
    IRQ15 : begin
              port[$A1]:=port[$A1] or $80;
            end;
  end;
END;
{----------------------------------------------------------}
FUNCTION Look_for_IK121(board:IK121_pointr):boolean;
  VAR
    buffer        : word;
    a,b           : boolean;
  BEGIN
    write_G26(board^.axis[1]^.baseadr,1,30,$0000);  (* 16-bit bus *)
    write_G26(board^.axis[2]^.baseadr,2,30,$0000);  (* 16-bit bus *)
    a:=true;
    b:=true;
    buffer:=read_g26(board^.axis[1]^.baseadr,1,28);
    buffer:=buffer and $FF00;
    if buffer<>$0800 then a:=false;
    buffer:=read_g26(board^.axis[2]^.baseadr,1,28);
    buffer:=buffer and $FF00;
    if buffer<>$0800 then b:=false;
    look_for_ik121:=a and b;
  END;
{----------------------------------------------------------}
 (*$I iic.pas*)
{----------------------------------------------------------}
{   Read/write init file                                   }
{----------------------------------------------------------}
PROCEDURE Write_Adr(address:word);
VAR
  initfile        : text;
  buffer          : string;
BEGIN
  Assign(initfile,'IK121.INI');
  (*$I-*)
  Rewrite(initfile);
  (*$I+*)
  if ioresult=0 then
    begin
      buffer:='base address axis 1 & 2 :';
      writeln(initfile,buffer);
      str(address,buffer);
      writeln(initfile,buffer);
      close(initfile);
    end;
END;
{----------------------------------------------------------}
FUNCTION Read_Adr:word;
VAR
  initfile          : text;
  buffer            : string;
  nofile            : boolean;
  code              : integer;
  address           : word;
BEGIN
  nofile:=true;
  Assign(initfile,'IK121.INI');
  (*$I-*)
  Reset(initfile);
  (*$I+*)
  if ioresult=0 then
    begin
      (*$I-*)
      readln(initfile,buffer);
      readln(initfile,buffer);
      (*$I+*)
      if (ioresult=0) then
        begin
          val(buffer,address,code);
          nofile:=false;
        end;
      close(initfile);
    end;
  if nofile or (code<>0) then
    begin
      address:=$0330;
      write_adr(address);
    end;
  read_adr:=address;
END;
{----------------------------------------------------------}
PROCEDURE Init_axis(axis:g26_pointr);
  BEGIN
    axis^.baseadr:=$0330;
    axis^.register0:=0;
    axis^.register1:=0;
    axis^.direction:=false;

    axis^.inimain.mode1024:=false;
    axis^.inimain.en_timer:=false;
    axis^.inimain.en_48bit:=true;
    axis^.inimain.onefold:=true;
    axis^.inimain.twofold:=false;
    axis^.inimain.fourfold:=false;
    axis^.inimain.arcmode:=false;
    axis^.inimain.arc180000:=false;
    axis^.inimain.sel_ri_1st:=false;
    axis^.inimain.sel_ri_2nd:=false;

    axis^.iniint.reg0:=false;
    axis^.iniint.reg1:=false;
    axis^.iniint.timer:=false;
    axis^.iniint.port:=false;

    axis^.inisync.en_l0_axis2:=false;
    axis^.inisync.en_latch2_axis2:=false;
    axis^.inisync.en_timer_axis2:=false;

    axis^.inilatch.en_l0_reg0:=false;
    axis^.inilatch.en_latch2_reg0:=false;
    axis^.inilatch.en_timer_reg0:=false;
    axis^.inilatch.en_l1_reg1:=false;
    axis^.inilatch.en_latch2_reg1:=false;
    axis^.inilatch.en_timer_reg1:=false;

    axis^.softcomm.start:=false;
    axis^.softcomm.stop:=true;
    axis^.softcomm.clear:=true;
    axis^.softcomm.latch0:=false;
    axis^.softcomm.latch1:=false;
    axis^.softcomm.latch2:=false;
    axis^.softcomm.clrfreq:=true;
    axis^.softcomm.clrstat:=true;

    axis^.refcomm.ristart:=false;
    axis^.refcomm.ristop:=false;
    axis^.refcomm.riclear:=false;
    axis^.refcomm.riclear2:=false;
    axis^.refcomm.rilatch:=false;
    axis^.refcomm.rilatch2:=false;

    axis^.cstatus.latch0:=false;
    axis^.cstatus.latch1:=false;
    axis^.cstatus.stop:=false;
    axis^.cstatus.sda:=false;
    axis^.cstatus.error_freq:=false;
    axis^.cstatus.ref_activ:=false;

    axis^.sstatus.ad00:=0;
    axis^.sstatus.ad90:=0;
    axis^.sstatus.amp_act:=3;
    axis^.sstatus.amp_min:=3;

    axis^.timer:=$FFFF;
    axis^.offset00:=0;
    axis^.offset90:=0;
  END;
{----------------------------------------------------------}
PROCEDURE Init_ik121(board:IK121_pointr);
  VAR
    error : boolean;
  BEGIN
    init_axis(board^.axis[1]);
    init_axis(board^.axis[2]);
    board^.axis[1]^.baseadr:=Read_Adr;
    board^.axis[2]^.baseadr:=board^.axis[1]^.baseadr;
    board^.axis[1]^.axis:=1;
    board^.axis[2]^.axis:=2;
    error:=false;
    Load_offset(board,error);
    Init_Handler(board^.axis[1]);
    Init_Handler(board^.axis[2]);
    Comm_Handler(board^.axis[1]);
    Comm_Handler(board^.axis[2]);
  END;
{----------------------------------------------------------}
END.
